from __future__ import annotations

from datetime import datetime, time as dtime
from typing import List, Optional

from bot.models import StandupConfig, StandupEntry
from bot.storage import BoardRepository, get_default_repo


class StandupService:
    def __init__(self, repo: BoardRepository | None = None) -> None:
        self.repo: BoardRepository = repo or get_default_repo()

    def configure(self, chat_id: int, enabled: bool,
                  time_of_day: Optional[dtime] = None,
                  questions: Optional[List[str]] = None) -> StandupConfig:
        board = self.repo.get_board(chat_id)
        cfg = board.get_or_create_standup_config()
        cfg.enabled = enabled
        if time_of_day is not None:
            cfg.time_of_day = time_of_day
        if questions is not None and questions:
            cfg.questions = questions
        board.standup_config = cfg
        self.repo.save_board(board)
        return cfg

    def add_answers(self, chat_id: int, user_id: int,
                    answers: List[str]) -> StandupEntry:
        board = self.repo.get_board(chat_id)
        entry = board.add_standup_entry(user_id=user_id, answers=answers,
                                        when=datetime.utcnow())
        self.repo.save_board(board)
        return entry

    def entries_for_day(self, chat_id: int, target_date) -> List[StandupEntry]:
        board = self.repo.get_board(chat_id)
        return [e for e in board.standup_entries if e.standup_date == target_date]
