from __future__ import annotations

import logging
import time
from typing import Any, Dict, Optional

from bot.handlers.commands import CommandHandler
from bot.services.bale_api import BaleAPI
from config.settings import get_settings
from utils.logging_utils import setup_logging


logger = logging.getLogger(__name__)


def _extract_update_offset(update: Dict[str, Any]) -> Optional[int]:
    if "update_id" in update:
        return int(update["update_id"])
    return None


def main() -> None:
    setup_logging()
    settings = get_settings()
    if not settings.bale_token:
        logger.error("BALE_TOKEN تنظیم نشده است. لطفاً در فایل .env مقدار آن را وارد کنید.")
        return

    logger.info("Bale Team Task Bot started.")
    api = BaleAPI()
    handler = CommandHandler(api=api)

    offset: Optional[int] = None
    while True:
        try:
            updates = api.get_updates(offset=offset, timeout=settings.long_poll_timeout)
            for upd in updates:
                handler.handle_update(upd)
                upo = _extract_update_offset(upd)
                if upo is not None:
                    offset = upo + 1
        except Exception as exc:  # noqa: BLE001
            logger.exception("Error while polling updates; retrying in 5 seconds...")
            time.sleep(5.0)


if __name__ == "__main__":
    main()
