import os, importlib, types

# main source folder
SRC_ROOT = os.path.join(os.path.dirname(os.path.dirname(__file__)), "src_original")

def _try_import():
    """
    Try to import main project module.
    Candidates: bot.py, main.py, app.py, server.py
    """
    import sys
    if SRC_ROOT not in sys.path:
        sys.path.insert(0, SRC_ROOT)

    candidates = ["bot", "main", "app", "server"]
    for name in candidates:
        try:
            return importlib.import_module(name)
        except Exception:
            continue
    return None

def _pick_handler(mod: types.ModuleType):
    """
    Pick a callable handler.
    Priority: handle_update(update, send) → handle_message(text, chat_id, send)
    """
    if not mod:
        return None, None

    if hasattr(mod, "handle_update") and callable(getattr(mod, "handle_update")):
        return "update", getattr(mod, "handle_update")

    if hasattr(mod, "handle_message") and callable(getattr(mod, "handle_message")):
        return "message", getattr(mod, "handle_message")

    return None, None

def _to_finglish(text: str) -> str:
    """
    Convert some Persian fixed phrases to Finglish.
    Extend as needed.
    """
    replacements = {
        "پیام شما دریافت شد.": "Payam shoma daryaft shod.",
        "Echo:": "Echo:",
        "دریافت شد": "Daryaft shod"
    }
    for k, v in replacements.items():
        text = text.replace(k, v)
    return text

def run(update: dict, send):
    """
    Main bridge: pass the update to the project handler.
    send(chat_id:int, text:str) is the function from routes.py
    """
    mod = _try_import()
    mode, handler = _pick_handler(mod)

    if handler is None:
        # fallback: simple echo
        msg = update.get("message") or update.get("edited_message")
        if msg and "text" in msg:
            chat_id = msg["chat"]["id"]
            text = _to_finglish(msg["text"])
            send(chat_id, f"Echo: {text}")
        return

    if mode == "update":
        return handler(update, lambda cid, txt, **kw: send(cid, _to_finglish(txt), **kw))

    if mode == "message":
        msg = update.get("message") or update.get("edited_message")
        if msg and "text" in msg:
            chat_id = msg["chat"]["id"]
            text = msg["text"]
            return handler(text, chat_id, lambda cid, txt, **kw: send(cid, _to_finglish(txt), **kw))

    # fallback for other message types
    msg = update.get("message") or update.get("edited_message")
    if msg:
        chat_id = msg.get("chat", {}).get("id")
        if chat_id:
            send(chat_id, _to_finglish("پیام شما دریافت شد."))
