from __future__ import annotations

import re
from typing import Any, Set

from aiogram import Router, F, types
from aiogram.fsm.context import FSMContext
from aiogram.filters import Command

from app.bots.hyper.states import LeadStates
from app.bots.hyper import keyboards
from app.bots.shared.api_client import BackendClient
from app.core.config import get_settings
from app.core.logging import get_logger

router = Router()
logger = get_logger("app.bots.hyper.account_purchase")
settings = get_settings()

MOBILE_RE = re.compile(r"^09\d{9}$")


def _parse_allowed_admins(raw: str | None) -> Set[int]:
    if not raw:
        return set()
    result: Set[int] = set()
    for chunk in raw.split(","):
        chunk = chunk.strip()
        if not chunk:
            continue
        try:
            result.add(int(chunk))
        except ValueError:
            continue
    return result


ALLOWED_ADMIN_IDS = _parse_allowed_admins(settings.HYPER_ADMIN_TELEGRAM_IDS)


# ---------------------------------------------------------------------------
# نقطه ورود – «📮 درخواست خرید حساب»
# ---------------------------------------------------------------------------


@router.message(F.text == "📮 درخواست خرید حساب")
async def lead_start(message: types.Message, state: FSMContext):
    """
    کاربر از منوی اصلی گزینه «📮 درخواست خرید حساب» را می‌زند.
    در این مرحله پلن مورد نظرش را انتخاب می‌کند.
    """
    await state.clear()
    await state.set_state(LeadStates.selecting_plan)
    await message.answer(
        "برای شروع، لطفاً پلن مورد نظر خود را انتخاب کنید:",
        reply_markup=keyboards.lead_plan_keyboard(),
    )


@router.callback_query(LeadStates.selecting_plan, F.data.startswith("lead:plan:"))
async def lead_plan_selected(callback: types.CallbackQuery, state: FSMContext):
    """
    پلن انتخاب شد → می‌رویم سراغ گرفتن شماره موبایل.
    """
    plan = callback.data.split(":")[-1]
    await state.update_data(selected_plan=plan)
    await state.set_state(LeadStates.waiting_phone)

    await callback.answer()
    await callback.message.answer(
        "شماره موبایل خود را به صورت ۱۱ رقمی و با 09 شروع ارسال کنید.\n"
        "مثال: 09123456789",
    )


@router.callback_query(F.data == "lead:cancel")
async def lead_cancel(callback: types.CallbackQuery, state: FSMContext):
    """
    لغو کل فرآیند خرید حساب.
    """
    await state.clear()
    await callback.answer()
    await callback.message.answer(
        "درخواست خرید حساب لغو شد.",
        reply_markup=keyboards.main_menu(),
    )


# ---------------------------------------------------------------------------
# مرحلهٔ گرفتن شماره موبایل
# ---------------------------------------------------------------------------


@router.message(LeadStates.waiting_phone)
async def lead_receive_phone(message: types.Message, state: FSMContext):
    """
    دریافت شماره موبایل از کاربر.
    پس از اعتبارسنجی، درخواست در بک‌اند ثبت و برای مدیران نوتیفیکیشن ارسال می‌شود.
    """
    phone = (message.text or "").strip()
    if not MOBILE_RE.match(phone):
        await message.answer(
            "شماره موبایل معتبر نیست.\n"
            "لطفاً شماره را به صورت ۱۱ رقمی و با 09 شروع ارسال کنید (مثال: 09123456789)."
        )
        return

    data = await state.get_data()
    plan = data.get("selected_plan", "base")

    tg_user = message.from_user
    payload: dict[str, Any] = {
        "telegram_id": tg_user.id,
        "telegram_username": tg_user.username,
        "full_name": tg_user.full_name,
        "phone": phone,
        "selected_plan": plan,
    }

    client = BackendClient()

    # تلاش برای ثبت در بک‌اند
    try:
        # نام مسیر را بر اساس API واقعی خودت اگر لازم بود تغییر بده
        resp = await client.post("/account-requests", payload)
        req_code = str(resp.get("code") or resp.get("id") or "")
    except Exception as exc:
        logger.exception("create account_request failed", exc_info=exc)
        await message.answer(
            "در ثبت درخواست مشکلی پیش آمد. لطفاً بعداً دوباره تلاش کنید "
            "یا با پشتیبانی در ارتباط باشید.",
            reply_markup=keyboards.main_menu(),
        )
        await state.clear()
        return

    # نوتیفای مدیران هایپر در تلگرام
    await _notify_admins_new_request(
        bot=message.bot,
        req_code=req_code,
        plan=plan,
        phone=phone,
        user=tg_user,
    )

    await state.set_state(LeadStates.waiting_payment_confirmation)
    await message.answer(
        "درخواست شما ثبت شد ✅\n"
        "به‌زودی لینک پرداخت توسط پشتیبانی برای شما ارسال می‌شود.\n"
        "بعد از انجام پرداخت، روی دکمه‌ی زیر بزنید.",
        reply_markup=keyboards.lead_wait_payment_keyboard(),
    )


async def _notify_admins_new_request(
    *,
    bot: types.Bot,
    req_code: str,
    plan: str,
    phone: str,
    user: types.User,
) -> None:
    """
    ارسال پیام به مدیران هایپر هنگام ثبت یک درخواست جدید.
    همراه با کیبورد اکشن‌های ادمین.
    """
    if not ALLOWED_ADMIN_IDS:
        return

    lines = [
        "📮 درخواست خرید حساب جدید دریافت شد:",
        f"پلن: {plan}",
        f"شماره تماس: {phone}",
        f"کاربر: {user.full_name} (@{user.username})",
        f"Telegram ID: {user.id}",
    ]
    if req_code:
        lines.append(f"کد درخواست: {req_code}")

    text = "\n".join(lines)

    for admin_id in ALLOWED_ADMIN_IDS:
        try:
            await bot.send_message(
                chat_id=admin_id,
                text=text,
                reply_markup=keyboards.admin_account_request_actions_keyboard(
                    user_id=user.id,
                    request_code=req_code or None,
                ),
            )
        except Exception as exc:
            logger.warning("failed to notify admin %s: %s", admin_id, exc)


# ---------------------------------------------------------------------------
# مرحله پرداخت – سمت کاربر
# ---------------------------------------------------------------------------


@router.callback_query(LeadStates.waiting_payment_confirmation, F.data == "lead:payment_done")
async def lead_payment_done(callback: types.CallbackQuery, state: FSMContext):
    """
    کاربر دکمه «✅ پرداخت انجام شد» را می‌زند.
    در اینجا فقط وضعیت را از دید کاربر ثبت می‌کنیم
    و پیام انتظار برای تایید نهایی را نمایش می‌دهیم.
    (تایید نهایی را ادمین از طریق پنل/بک‌اند انجام می‌دهد.)
    """
    await callback.answer("درخواست بررسی پرداخت ثبت شد ✅", show_alert=True)
    await state.clear()
    await callback.message.answer(
        "با تشکر از پرداخت شما 🙏\n"
        "در حال بررسی تراکنش هستیم. پس از تأیید، اطلاعات حساب برای شما ارسال خواهد شد.",
        reply_markup=keyboards.main_menu(),
    )


# ---------------------------------------------------------------------------
# اکشن‌های ادمین روی درخواست‌ها (از نوتیفیکیشن تلگرام)
# ---------------------------------------------------------------------------


@router.callback_query(F.data.startswith("leadadm:paid:"))
async def admin_mark_paid(callback: types.CallbackQuery, state: FSMContext):
    """
    ادمین روی «✅ پرداخت تایید شد» می‌زند.
    - می‌توانیم بک‌اند را هم آپدیت کنیم (اختیاری، بسته به API واقعی).
    - سپس به کاربر پیام تایید ارسال می‌شود.
    """
    _, _, user_id_str = callback.data.split(":", 2)
    try:
        user_id = int(user_id_str)
    except ValueError:
        await callback.answer("شناسه کاربر نامعتبر است.", show_alert=True)
        return

    # این بخش را بر اساس API واقعی می‌توانی فعال/تکمیل کنی:
    try:
        client = BackendClient()
        # مثال: await client.post("/account-requests/mark-paid", {"telegram_id": user_id})
    except Exception as exc:
        logger.warning("mark-paid backend call failed: %s", exc)

    # پیام به خود کاربر
    try:
        await callback.bot.send_message(
            chat_id=user_id,
            text=(
                "پرداخت شما تأیید شد ✅\n"
                "حساب بازاریا شما به‌زودی فعال می‌شود و اطلاعات ورود برایتان ارسال خواهد شد."
            ),
        )
    except Exception as exc:
        logger.warning("failed to notify user %s about paid status: %s", user_id, exc)

    await callback.answer("پیام تأیید برای کاربر ارسال شد.", show_alert=True)


@router.callback_query(F.data.startswith("leadadm:reject:"))
async def admin_reject_request(callback: types.CallbackQuery, state: FSMContext):
    """
    ادمین روی «❌ رد درخواست» می‌زند.
    """
    _, _, user_id_str = callback.data.split(":", 2)
    try:
        user_id = int(user_id_str)
    except ValueError:
        await callback.answer("شناسه کاربر نامعتبر است.", show_alert=True)
        return

    try:
        client = BackendClient()
        # مثال: await client.post("/account-requests/reject", {"telegram_id": user_id})
    except Exception as exc:
        logger.warning("reject backend call failed: %s", exc)

    try:
        await callback.bot.send_message(
            chat_id=user_id,
            text=(
                "درخواست خرید حساب شما توسط پشتیبانی رد شد.\n"
                "در صورت نیاز می‌توانید با پشتیبانی در ارتباط باشید."
            ),
        )
    except Exception as exc:
        logger.warning("failed to notify user %s about rejection: %s", user_id, exc)

    await callback.answer("درخواست کاربر رد شد و به او اطلاع داده شد.", show_alert=True)
