# file: negarprint/web_admin.py
from __future__ import annotations

import math
import os
from collections import defaultdict
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional

from flask import (
    Flask,
    flash,
    redirect,
    render_template_string,
    request,
    session,
    url_for,
)

from .config import load_settings
from .database import Database


# -------------------------------
# تنظیمات و اتصال به دیتابیس
# -------------------------------
settings = load_settings()
db = Database(settings.database_path)

app = Flask(__name__)
app.secret_key = os.environ.get("ADMIN_WEB_SECRET", "change-me")
ADMIN_PASSWORD = os.environ.get("ADMIN_WEB_PASSWORD", "")


# -------------------------------
# توابع کمکی
# -------------------------------
def is_logged_in() -> bool:
    return session.get("admin_logged_in") is True


def parse_created_at(order: Dict[str, Any]) -> Optional[datetime]:
    created_at = order.get("created_at")
    if isinstance(created_at, datetime):
        return created_at
    if isinstance(created_at, str):
        s = created_at[:19]
        for fmt in ("%Y-%m-%d %H:%M:%S", "%Y-%m-%dT%H:%M:%S", "%Y-%m-%d"):
            try:
                return datetime.strptime(s, fmt)
            except Exception:
                continue
    return None


def _estimate_order_price(db: Database, data: Dict[str, Any]) -> Optional[int]:
    """
    همان منطق قیمت‌گذاری که در bot.py استفاده شده است (کپی شده).
    براساس تعرفه‌ها و تعداد صفحات، قیمت هر سفارش را تخمین می‌زند.
    """
    files = data.get("files") or []
    if not isinstance(files, list):
        return None

    total = 0
    had_any_price = False

    for f in files:
        try:
            if f.get("selected_pages"):
                pages = len(f["selected_pages"])
            else:
                pages = int(f.get("page_count") or 0)
            if pages <= 0:
                continue

            duplex = f.get("duplex") or "single"
            copies = int(f.get("copies") or 1)
            paper_size = f.get("paper_size") or "A4"
            color = f.get("color") or "bw"
            paper_type = f.get("paper_type") or "plain"
            file_kind = f.get("file_kind") or "print"
            pages_per_sheet = int(f.get("pages_per_sheet") or 1)
            if pages_per_sheet <= 0:
                pages_per_sheet = 1

            sheets_per_copy = math.ceil(pages / pages_per_sheet)
            sheets = sheets_per_copy * copies

            tariff = db.get_tariff(
                file_kind=file_kind,
                paper_size=paper_size,
                color=color,
                duplex=duplex,
                paper_type=paper_type,
            )
            if not tariff:
                continue

            had_any_price = True
            price_per_sheet = int(tariff["price_per_sheet"])
            total += sheets * price_per_sheet
        except Exception:
            continue

    return total if had_any_price else None


def order_matches_filters(
    order: Dict[str, Any],
    status_filter: Optional[str],
    q: str,
    date_from: Optional[datetime],
    date_to: Optional[datetime],
) -> bool:
    # وضعیت
    if status_filter and order.get("status") != status_filter:
        return False

    # بازه‌ی تاریخ
    dt = parse_created_at(order)
    if date_from and (dt is None or dt < date_from):
        return False
    if date_to and (dt is None or dt > date_to):
        return False

    # جستجو
    if q:
        q_lower = q.strip().lower()
        data = order.get("data") or {}
        profile = data.get("profile") or {}
        name = str(profile.get("name", "")).lower()
        phone = str(profile.get("phone", "")).lower()
        uid = str(order.get("user_id", "")).lower()
        if q_lower not in name and q_lower not in phone and q_lower not in uid:
            return False

    return True


def compute_financial_summary(db: Database, orders: List[Dict[str, Any]]) -> Dict[str, Any]:
    """
    گزارش مالی ساده بر اساس سفارش‌های انجام‌شده (status='done')
    """
    total_done = 0.0
    count_done = 0
    per_day: Dict[str, float] = defaultdict(float)

    for o in orders:
        if o.get("status") != "done":
            continue

        amount = o.get("amount")
        if amount is None:
            data = o.get("data") or {}
            amount = _estimate_order_price(db, data) or 0

        try:
            amount_f = float(amount)
        except Exception:
            continue

        if amount_f <= 0:
            continue

        total_done += amount_f
        count_done += 1

        dt = parse_created_at(o)
        if dt:
            key = dt.date().isoformat()
            per_day[key] += amount_f

    avg_ticket = total_done / count_done if count_done else 0.0

    today = datetime.utcnow().date()
    last_7_days = []
    for i in range(6, -1, -1):
        d = today - timedelta(days=i)
        iso = d.isoformat()
        last_7_days.append(
            {"date": iso, "amount": per_day.get(iso, 0.0)}
        )

    return {
        "total_done": total_done,
        "count_done": count_done,
        "avg_ticket": avg_ticket,
        "last_7_days": last_7_days,
    }


# -------------------------------
# Middleware: نیاز به لاگین
# -------------------------------
@app.before_request
def _require_login() -> None:
    public_paths = ("/admin/login", "/static", "/favicon.ico")
    if any(request.path.startswith(p) for p in public_paths):
        return

    if request.path.startswith("/admin"):
        if not is_logged_in():
            next_url = request.path
            if request.query_string:
                next_url = f"{next_url}?{request.query_string.decode()}"
            return redirect(url_for("admin_login", next=next_url))


# -------------------------------
# قالب‌ها
# -------------------------------
LOGIN_TEMPLATE = """
<!doctype html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>ورود به پنل مدیریت</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body {
      direction: rtl;
      text-align: right;
      background-color: #f5f5f5;
    }
    .login-card {
      max-width: 400px;
      margin: 80px auto;
    }
  </style>
</head>
<body>
<div class="container">
  <div class="card login-card shadow-sm">
    <div class="card-body">
      <h4 class="card-title mb-3">ورود به پنل مدیریت</h4>
      {% with messages = get_flashed_messages() %}
        {% if messages %}
          <div class="alert alert-warning">
            {% for m in messages %}{{ m }}<br>{% endfor %}
          </div>
        {% endif %}
      {% endwith %}
      {% if error %}
        <div class="alert alert-danger">{{ error }}</div>
      {% endif %}
      <form method="post">
        <div class="mb-3">
          <label class="form-label">رمز عبور</label>
          <input type="password" name="password" class="form-control" required autofocus>
        </div>
        <button type="submit" class="btn btn-primary w-100">ورود</button>
      </form>
    </div>
  </div>
</div>
</body>
</html>
"""


ADMIN_TEMPLATE = """
<!doctype html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>پنل مدیریت NegarPrint</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    body {
      direction: rtl;
      text-align: right;
    }
    .table td, .table th {
      vertical-align: middle;
      font-size: 0.85rem;
    }
    .badge-status-new {
      background-color: #fff3cd;
      color: #856404;
    }
    .badge-status-done {
      background-color: #d4edda;
      color: #155724;
    }
  </style>
</head>
<body class="bg-light">
<nav class="navbar navbar-expand-lg navbar-dark bg-dark mb-3">
  <div class="container-fluid">
    <a class="navbar-brand" href="{{ url_for('admin_index') }}">NegarPrint Admin</a>
    <div class="d-flex">
      <span class="navbar-text me-3">ادمین</span>
      <a href="{{ url_for('admin_logout') }}" class="btn btn-sm btn-outline-light">خروج</a>
    </div>
  </div>
</nav>

<div class="container mb-4">
  <ul class="nav nav-tabs mb-3">
    <li class="nav-item">
      <a class="nav-link {% if tab == 'orders' %}active{% endif %}" href="{{ url_for('admin_index', tab='orders') }}">سفارش‌ها</a>
    </li>
    <li class="nav-item">
      <a class="nav-link {% if tab == 'tariffs' %}active{% endif %}" href="{{ url_for('admin_index', tab='tariffs') }}">تعرفه‌ها</a>
    </li>
    <li class="nav-item">
      <a class="nav-link {% if tab == 'staff' %}active{% endif %}" href="{{ url_for('admin_index', tab='staff') }}">تیم</a>
    </li>
    <li class="nav-item">
      <a class="nav-link {% if tab == 'reports' %}active{% endif %}" href="{{ url_for('admin_index', tab='reports') }}">گزارش مالی</a>
    </li>
  </ul>

  {% with messages = get_flashed_messages() %}
    {% if messages %}
      <div class="alert alert-info">
        {% for m in messages %}{{ m }}<br>{% endfor %}
      </div>
    {% endif %}
  {% endwith %}

  {# ------------------ TAB: ORDERS ------------------ #}
  {% if tab == 'orders' %}

    <div class="card mb-3">
      <div class="card-body">
        <h5 class="card-title mb-3">فیلتر سفارش‌ها</h5>
        <form class="row g-2" method="get">
          <input type="hidden" name="tab" value="orders">
          <div class="col-md-2">
            <label class="form-label">وضعیت</label>
            <select name="status" class="form-select form-select-sm">
              <option value="">همه</option>
              <option value="new"  {% if status_filter == 'new' %}selected{% endif %}>جدید</option>
              <option value="done" {% if status_filter == 'done' %}selected{% endif %}>انجام‌شده</option>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">جستجو (نام / تلفن / User ID)</label>
            <input type="text" name="q" value="{{ q }}" class="form-control form-control-sm">
          </div>
          <div class="col-md-2">
            <label class="form-label">از تاریخ</label>
            <input type="date" name="date_from" value="{{ date_from_str }}" class="form-control form-control-sm">
          </div>
          <div class="col-md-2">
            <label class="form-label">تا تاریخ</label>
            <input type="date" name="date_to" value="{{ date_to_str }}" class="form-control form-control-sm">
          </div>
          <div class="col-md-3 d-flex align-items-end justify-content-end">
            <button type="submit" class="btn btn-primary btn-sm me-2">اعمال فیلتر</button>
            <a href="{{ url_for('admin_index', tab='orders') }}" class="btn btn-outline-secondary btn-sm">حذف فیلتر</a>
          </div>
        </form>
      </div>
    </div>

    <div class="card mb-3">
      <div class="card-body">
        <h5 class="card-title mb-3">لیست سفارش‌ها ({{ orders|length }} مورد)</h5>
        <div class="table-responsive">
          <table class="table table-sm table-striped table-hover align-middle">
            <thead>
            <tr>
              <th>#</th>
              <th>وضعیت</th>
              <th>User ID</th>
              <th>نام</th>
              <th>تلفن</th>
              <th>تعداد فایل</th>
              <th>مبلغ تخمینی</th>
              <th>تاریخ ثبت</th>
              <th>عملیات</th>
            </tr>
            </thead>
            <tbody>
            {% if orders %}
              {% for o in orders %}
                {% set data = o.data or {} %}
                {% set profile = data.get('profile') or {} %}
                {% set files = data.get('files') or [] %}
                <tr>
                  <td>{{ o.id }}</td>
                  <td>
                    {% if o.status == 'new' %}
                      <span class="badge badge-status-new">جدید</span>
                    {% elif o.status == 'done' %}
                      <span class="badge badge-status-done">انجام‌شده</span>
                    {% else %}
                      <span class="badge bg-secondary">{{ o.status }}</span>
                    {% endif %}
                  </td>
                  <td>{{ o.user_id }}</td>
                  <td>{{ profile.get('name', '-') }}</td>
                  <td>{{ profile.get('phone', '-') }}</td>
                  <td>{{ files|length }}</td>
                  <td>
                    {% if o.amount is not none %}
                      {{ "%.0f"|format(o.amount) }}
                    {% else %}
                      -
                    {% endif %}
                  </td>
                  <td>{{ o.created_at }}</td>
                  <td>
                    <button
                      type="button"
                      class="btn btn-sm btn-info mb-1"
                      data-bs-toggle="modal"
                      data-bs-target="#orderDetailModal"
                      data-order-id="{{ o.id }}"
                    >
                      جزئیات
                    </button>

                    {% if o.status != 'done' %}
                      <form method="post" class="d-inline">
                        <input type="hidden" name="action" value="set_order_status">
                        <input type="hidden" name="order_id" value="{{ o.id }}">
                        <input type="hidden" name="status" value="done">
                        <input type="hidden" name="tab" value="orders">
                        <input type="hidden" name="status_filter" value="{{ status_filter or '' }}">
                        <input type="hidden" name="q" value="{{ q or '' }}">
                        <input type="hidden" name="date_from" value="{{ date_from_str or '' }}">
                        <input type="hidden" name="date_to" value="{{ date_to_str or '' }}">
                        <button type="submit" class="btn btn-sm btn-success mb-1">انجام شد</button>
                      </form>
                    {% endif %}
                    {% if o.status != 'new' %}
                      <form method="post" class="d-inline">
                        <input type="hidden" name="action" value="set_order_status">
                        <input type="hidden" name="order_id" value="{{ o.id }}">
                        <input type="hidden" name="status" value="new">
                        <input type="hidden" name="tab" value="orders">
                        <input type="hidden" name="status_filter" value="{{ status_filter or '' }}">
                        <input type="hidden" name="q" value="{{ q or '' }}">
                        <input type="hidden" name="date_from" value="{{ date_from_str or '' }}">
                        <input type="hidden" name="date_to" value="{{ date_to_str or '' }}">
                        <button type="submit" class="btn btn-sm btn-outline-secondary mb-1">برگرد به جدید</button>
                      </form>
                    {% endif %}
                  </td>
                </tr>
              {% endfor %}
            {% else %}
              <tr><td colspan="9" class="text-center">سفارشی یافت نشد.</td></tr>
            {% endif %}
            </tbody>
          </table>
        </div>
      </div>
    </div>

    {# قالب‌های مخفی برای Modal جزئیات سفارش #}
    <div id="order-detail-templates" class="d-none">
      {% for o in orders %}
        {% set data = o.data or {} %}
        {% set profile = data.get('profile') or {} %}
        {% set files = data.get('files') or [] %}
        <div id="order-detail-{{ o.id }}">
          <h6 class="mb-2">سفارش #{{ o.id }}</h6>
          <p class="mb-1"><strong>وضعیت:</strong>
            {% if o.status == 'new' %}جدید{% elif o.status == 'done' %}انجام‌شده{% else %}{{ o.status }}{% endif %}
          </p>
          <p class="mb-1"><strong>User ID:</strong> {{ o.user_id }}</p>
          <p class="mb-1"><strong>نام:</strong> {{ profile.get('name','-') }}</p>
          <p class="mb-1"><strong>تلفن:</strong> {{ profile.get('phone','-') }}</p>
          <p class="mb-1"><strong>یوزرنیم:</strong> {{ profile.get('username','-') }}</p>
          <p class="mb-1"><strong>تاریخ ثبت:</strong> {{ o.created_at }}</p>
          <p class="mb-2">
            <strong>مبلغ تخمینی سفارش:</strong>
            {% if o.amount is not none %}
              {{ "%.0f"|format(o.amount) }}
            {% else %}
              -
            {% endif %}
          </p>
          <hr>
          <h6 class="mb-2">فایل‌ها ({{ files|length }})</h6>
          {% if files %}
            <ol class="mb-0">
              {% for f in files %}
                {% set page_desc = '' %}
                {% if f.selected_pages %}
                  {% set page_desc = (f.selected_pages|length|string) + " صفحه انتخابی" %}
                {% elif f.page_count %}
                  {% set page_desc = f.page_count|string + " صفحه" %}
                {% endif %}
                <li class="mb-2">
                  <div><strong>نوع فایل:</strong> {{ f.file_kind or 'print' }}</div>
                  <div><strong>سایز کاغذ:</strong> {{ f.paper_size or 'A4' }}</div>
                  <div><strong>نوع چاپ:</strong> {{ f.color or 'bw' }}</div>
                  <div><strong>تک‌رو/دو‌رو:</strong> {% if f.duplex == 'single' %}تک‌رو{% else %}پشت‌ورو{% endif %}</div>
                  <div><strong>نوع کاغذ:</strong> {{ f.paper_type or '-' }}</div>
                  <div><strong>نسخه:</strong> {{ f.copies or 1 }}</div>
                  <div><strong>صفحات در هر برگ:</strong> {{ f.pages_per_sheet or 1 }}</div>
                  {% if page_desc %}
                    <div><strong>صفحات:</strong> {{ page_desc }}</div>
                  {% endif %}
                </li>
              {% endfor %}
            </ol>
          {% else %}
            <p class="mb-0 text-muted">فایلی ثبت نشده است.</p>
          {% endif %}
        </div>
      {% endfor %}
    </div>

  {# ------------------ TAB: TARIFFS ------------------ #}
  {% elif tab == 'tariffs' %}

    <div class="row">
      <div class="col-lg-8">
        <div class="card mb-3">
          <div class="card-body">
            <h5 class="card-title mb-3">تعرفه‌های موجود</h5>
            <div class="table-responsive">
              <table class="table table-sm table-striped">
                <thead>
                <tr>
                  <th>#</th>
                  <th>نوع فایل</th>
                  <th>سایز</th>
                  <th>رنگ</th>
                  <th>تک‌رو/دو‌رو</th>
                  <th>نوع کاغذ</th>
                  <th>قیمت هر برگ</th>
                  <th>تاریخ</th>
                </tr>
                </thead>
                <tbody>
                {% if tariffs %}
                  {% for t in tariffs %}
                    <tr>
                      <td>{{ t.id }}</td>
                      <td>{{ t.file_kind }}</td>
                      <td>{{ t.paper_size }}</td>
                      <td>{{ t.color }}</td>
                      <td>{{ t.duplex }}</td>
                      <td>{{ t.paper_type }}</td>
                      <td>{{ t.price_per_sheet }}</td>
                      <td>{{ t.created_at }}</td>
                    </tr>
                  {% endfor %}
                {% else %}
                  <tr><td colspan="8" class="text-center">تعرفه‌ای ثبت نشده است.</td></tr>
                {% endif %}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      </div>

      <div class="col-lg-4">
        <div class="card mb-3">
          <div class="card-body">
            <h5 class="card-title mb-2">ثبت / به‌روزرسانی تعرفه</h5>
            <p class="text-muted small">
              اگر ترکیب مشخصات از قبل وجود داشته باشد، فقط قیمت به‌روزرسانی می‌شود.
            </p>
            <form method="post">
              <input type="hidden" name="action" value="upsert_tariff">
              <input type="hidden" name="tab" value="tariffs">

              <div class="mb-2">
                <label class="form-label">نوع فایل</label>
                <input type="text" name="file_kind" class="form-control form-control-sm" required>
              </div>
              <div class="mb-2">
                <label class="form-label">سایز کاغذ</label>
                <input type="text" name="paper_size" class="form-control form-control-sm" required>
              </div>
              <div class="mb-2">
                <label class="form-label">رنگ</label>
                <input type="text" name="color" class="form-control form-control-sm" required>
              </div>
              <div class="mb-2">
                <label class="form-label">تک‌رو/دو‌رو</label>
                <input type="text" name="duplex" class="form-control form-control-sm" required>
              </div>
              <div class="mb-2">
                <label class="form-label">نوع کاغذ</label>
                <input type="text" name="paper_type" class="form-control form-control-sm" required>
              </div>
              <div class="mb-3">
                <label class="form-label">قیمت هر برگ</label>
                <input type="number" name="price_per_sheet" class="form-control form-control-sm" required>
              </div>
              <button type="submit" class="btn btn-primary btn-sm w-100">ثبت / به‌روزرسانی</button>
            </form>
          </div>
        </div>
      </div>
    </div>

  {# ------------------ TAB: STAFF ------------------ #}
  {% elif tab == 'staff' %}

    <div class="row">
      <div class="col-lg-8">
        <div class="card mb-3">
          <div class="card-body">
            <h5 class="card-title mb-3">لیست اعضای تیم</h5>
            <div class="table-responsive">
              <table class="table table-sm table-striped align-middle">
                <thead>
                <tr>
                  <th>#</th>
                  <th>User ID</th>
                  <th>نقش</th>
                  <th>نام کاربری</th>
                  <th>نام کامل</th>
                  <th>وضعیت</th>
                  <th>شیفت</th>
                  <th>عملیات</th>
                </tr>
                </thead>
                <tbody>
                {% if staff %}
                  {% for s in staff %}
                    <tr>
                      <td>{{ s.id }}</td>
                      <td>{{ s.user_id }}</td>
                      <td>{{ s.role }}</td>
                      <td>{{ s.username or '-' }}</td>
                      <td>{{ s.full_name or '-' }}</td>
                      <td>
                        {% if s.active %}
                          <span class="badge bg-success">فعال</span>
                        {% else %}
                          <span class="badge bg-secondary">غیرفعال</span>
                        {% endif %}
                      </td>
                      <td>
                        {% if s.role == 'operator' %}
                          {% if s.on_duty %}
                            <span class="badge bg-info text-dark">روی شیفت</span>
                          {% else %}
                            <span class="badge bg-light text-dark border">خارج از شیفت</span>
                          {% endif %}
                        {% else %}
                          -
                        {% endif %}
                      </td>
                      <td>
                        <form method="post" class="d-inline">
                          <input type="hidden" name="action" value="toggle_staff_active">
                          <input type="hidden" name="user_id" value="{{ s.user_id }}">
                          <input type="hidden" name="role" value="{{ s.role }}">
                          <input type="hidden" name="current_active" value="{{ 1 if s.active else 0 }}">
                          <input type="hidden" name="tab" value="staff">
                          {% if s.active %}
                            <button type="submit" class="btn btn-sm btn-outline-secondary">غیرفعال</button>
                          {% else %}
                            <button type="submit" class="btn btn-sm btn-success">فعال</button>
                          {% endif %}
                        </form>

                        {% if s.role == 'operator' %}
                          <form method="post" class="d-inline">
                            <input type="hidden" name="action" value="toggle_staff_duty">
                            <input type="hidden" name="user_id" value="{{ s.user_id }}">
                            <input type="hidden" name="role" value="{{ s.role }}">
                            <input type="hidden" name="current_on_duty" value="{{ 1 if s.on_duty else 0 }}">
                            <input type="hidden" name="tab" value="staff">
                            {% if s.on_duty %}
                              <button type="submit" class="btn btn-sm btn-outline-warning">خارج از شیفت</button>
                            {% else %}
                              <button type="submit" class="btn btn-sm btn-warning">روی شیفت</button>
                            {% endif %}
                          </form>
                        {% endif %}

                        <form method="post" class="d-inline" onsubmit="return confirm('حذف این کاربر از تیم؟');">
                          <input type="hidden" name="action" value="remove_staff">
                          <input type="hidden" name="user_id" value="{{ s.user_id }}">
                          <input type="hidden" name="role" value="{{ s.role }}">
                          <input type="hidden" name="tab" value="staff">
                          <button type="submit" class="btn btn-sm btn-outline-danger">حذف</button>
                        </form>
                      </td>
                    </tr>
                  {% endfor %}
                {% else %}
                  <tr><td colspan="8" class="text-center">عضوی ثبت نشده است.</td></tr>
                {% endif %}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      </div>

      <div class="col-lg-4">
        <div class="card mb-3">
          <div class="card-body">
            <h5 class="card-title mb-2">افزودن عضو جدید</h5>
            <form method="post">
              <input type="hidden" name="action" value="add_staff">
              <input type="hidden" name="tab" value="staff">

              <div class="mb-2">
                <label class="form-label">User ID (تلگرام)</label>
                <input type="number" name="user_id" class="form-control form-control-sm" required>
              </div>
              <div class="mb-2">
                <label class="form-label">نقش</label>
                <select name="role" class="form-select form-select-sm" required>
                  <option value="operator">operator</option>
                  <option value="admin">admin</option>
                  <option value="owner">owner</option>
                </select>
              </div>
              <div class="mb-2">
                <label class="form-label">نام کاربری (بدون @)</label>
                <input type="text" name="username" class="form-control form-control-sm">
              </div>
              <div class="mb-3">
                <label class="form-label">نام کامل</label>
                <input type="text" name="full_name" class="form-control form-control-sm">
              </div>
              <button type="submit" class="btn btn-primary btn-sm w-100">افزودن</button>
            </form>
          </div>
        </div>
      </div>
    </div>

  {# ------------------ TAB: REPORTS ------------------ #}
  {% elif tab == 'reports' %}

    <div class="row">
      <div class="col-lg-5">
        <div class="card mb-3">
          <div class="card-body">
            <h5 class="card-title mb-3">خلاصه‌ی مالی</h5>
            <p class="mb-1">تعداد سفارش‌های انجام‌شده: <strong>{{ financial.count_done }}</strong></p>
            <p class="mb-1">جمع مبلغ سفارش‌های انجام‌شده: <strong>{{ "%.0f"|format(financial.total_done) }}</strong></p>
            <p class="mb-1">میانگین مبلغ هر سفارش (انجام‌شده): <strong>{{ "%.0f"|format(financial.avg_ticket) }}</strong></p>
          </div>
        </div>
      </div>

      <div class="col-lg-7">
        <div class="card mb-3">
          <div class="card-body">
            <h5 class="card-title mb-3">گردش ۷ روز اخیر (فقط انجام‌شده)</h5>
            <canvas id="revChart" height="120"></canvas>
          </div>
        </div>
      </div>
    </div>

    <script>
      document.addEventListener("DOMContentLoaded", function () {
        var ctx = document.getElementById("revChart");
        if (!ctx) return;
        var rows = {{ financial.last_7_days | tojson }};
        var labels = rows.map(function (r) { return r.date; });
        var values = rows.map(function (r) { return r.amount; });
        new Chart(ctx, {
          type: "bar",
          data: {
            labels: labels,
            datasets: [{
              label: "مبلغ سفارش‌های انجام‌شده",
              data: values
            }]
          },
          options: {
            responsive: true,
            scales: {
              y: {
                beginAtZero: true
              }
            }
          }
        });
      });
    </script>

  {% endif %}
</div>

{# Modal جزئیات سفارش #}
<div class="modal fade" id="orderDetailModal" tabindex="-1" aria-labelledby="orderDetailModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="orderDetailModalLabel">جزئیات سفارش</h5>
        <button type="button" class="btn-close ms-0" data-bs-dismiss="modal" aria-label="بستن"></button>
      </div>
      <div class="modal-body">
        <div id="orderDetailBody">در حال بارگذاری...</div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">بستن</button>
      </div>
    </div>
  </div>
</div>

<script>
  var detailModal = document.getElementById("orderDetailModal");
  if (detailModal) {
    detailModal.addEventListener("show.bs.modal", function (event) {
      var button = event.relatedTarget;
      if (!button) return;
      var orderId = button.getAttribute("data-order-id");
      var tmpl = document.getElementById("order-detail-" + orderId);
      var body = document.getElementById("orderDetailBody");
      var title = document.getElementById("orderDetailModalLabel");
      if (title) {
        title.textContent = "جزئیات سفارش #" + orderId;
      }
      if (body) {
        if (tmpl) {
          body.innerHTML = tmpl.innerHTML;
        } else {
          body.textContent = "جزئیات در دسترس نیست.";
        }
      }
    });
  }
</script>

</body>
</html>
"""


# -------------------------------
# Routes
# -------------------------------
@app.route("/admin/login", methods=["GET", "POST"])
def admin_login():
    error = None
    if request.method == "POST":
        password = request.form.get("password") or ""
        if ADMIN_PASSWORD and password == ADMIN_PASSWORD:
            session["admin_logged_in"] = True
            flash("با موفقیت وارد شدید.")
            next_url = request.args.get("next") or url_for("admin_index")
            return redirect(next_url)
        else:
            error = "رمز عبور نادرست است."
    return render_template_string(LOGIN_TEMPLATE, error=error)


@app.route("/admin/logout")
def admin_logout():
    session.clear()
    flash("خارج شدید.")
    return redirect(url_for("admin_login"))


@app.route("/admin", methods=["GET", "POST"])
def admin_index():
    tab = request.args.get("tab", "orders")

    # -------------- POST: اعمال عملیات --------------
    if request.method == "POST":
        action = request.form.get("action") or ""
        tab = request.form.get("tab", tab)

        if action == "set_order_status":
            try:
                order_id = int(request.form.get("order_id") or "0")
            except ValueError:
                order_id = 0
            status = request.form.get("status") or ""
            if order_id and status:
                db.set_order_status(order_id, status)
                flash("وضعیت سفارش به‌روزرسانی شد.")

            status_filter = request.form.get("status_filter") or ""
            q = request.form.get("q") or ""
            date_from = request.form.get("date_from") or ""
            date_to = request.form.get("date_to") or ""
            return redirect(
                url_for(
                    "admin_index",
                    tab="orders",
                    status=status_filter or None,
                    q=q or None,
                    date_from=date_from or None,
                    date_to=date_to or None,
                )
            )

        elif action == "upsert_tariff":
            file_kind = (request.form.get("file_kind") or "").strip()
            paper_size = (request.form.get("paper_size") or "").strip()
            color = (request.form.get("color") or "").strip()
            duplex = (request.form.get("duplex") or "").strip()
            paper_type = (request.form.get("paper_type") or "").strip()
            try:
                price_per_sheet = int(request.form.get("price_per_sheet") or "0")
            except ValueError:
                price_per_sheet = 0

            if all([file_kind, paper_size, color, duplex, paper_type]) and price_per_sheet > 0:
                db.upsert_tariff(
                    file_kind=file_kind,
                    paper_size=paper_size,
                    color=color,
                    duplex=duplex,
                    paper_type=paper_type,
                    price_per_sheet=price_per_sheet,
                )
                flash("تعرفه ثبت / به‌روزرسانی شد.")
            else:
                flash("ورودی تعرفه ناقص یا نامعتبر است.", "warning")

            return redirect(url_for("admin_index", tab="tariffs"))

        elif action == "add_staff":
            try:
                user_id = int(request.form.get("user_id") or "0")
            except ValueError:
                user_id = 0
            role = (request.form.get("role") or "").strip()
            username = (request.form.get("username") or "").strip() or None
            full_name = (request.form.get("full_name") or "").strip() or None

            if user_id and role:
                db.add_staff(user_id=user_id, role=role, username=username, full_name=full_name)
                flash("عضو جدید به تیم افزوده شد.")
            else:
                flash("اطلاعات عضو جدید ناقص است.", "warning")

            return redirect(url_for("admin_index", tab="staff"))

        elif action == "toggle_staff_active":
            try:
                user_id = int(request.form.get("user_id") or "0")
            except ValueError:
                user_id = 0
            role = (request.form.get("role") or "").strip()
            current_active = request.form.get("current_active") == "1"
            if user_id and role:
                db.set_active(user_id, role, not current_active)
                flash("وضعیت فعال/غیرفعال تغییر کرد.")
            return redirect(url_for("admin_index", tab="staff"))

        elif action == "toggle_staff_duty":
            try:
                user_id = int(request.form.get("user_id") or "0")
            except ValueError:
                user_id = 0
            role = (request.form.get("role") or "").strip()
            current_on_duty = request.form.get("current_on_duty") == "1"
            if user_id and role:
                db.set_duty(user_id, not current_on_duty, role)
                flash("وضعیت شیفت اپراتور تغییر کرد.")
            return redirect(url_for("admin_index", tab="staff"))

        elif action == "remove_staff":
            try:
                user_id = int(request.form.get("user_id") or "0")
            except ValueError:
                user_id = 0
            role = (request.form.get("role") or "").strip() or None
            if user_id:
                db.remove_staff(user_id, role=role)
                flash("عضو از تیم حذف شد.")
            return redirect(url_for("admin_index", tab="staff"))

    # -------------- GET: آماده‌سازی داده‌ها برای نمایش --------------

    # همه‌ی سفارش‌ها (برای فیلتر و گزارش)
    orders_all: List[Dict[str, Any]] = db.list_orders(statuses=None, limit=1000)

    # محاسبه مبلغ تخمینی برای هر سفارش
    for o in orders_all:
        data = o.get("data") or {}
        amount = o.get("price_cached")
        if amount is None:
            amount = _estimate_order_price(db, data)
        o["amount"] = amount

    # فیلترها برای تب سفارش‌ها
    status_filter = request.args.get("status") or ""
    q = request.args.get("q") or ""
    date_from_str = request.args.get("date_from") or ""
    date_to_str = request.args.get("date_to") or ""

    date_from = None
    date_to = None
    if date_from_str:
        try:
            date_from = datetime.strptime(date_from_str, "%Y-%m-%d")
        except Exception:
            date_from = None
    if date_to_str:
        try:
            date_to = datetime.strptime(date_to_str, "%Y-%m-%d") + timedelta(days=1)
        except Exception:
            date_to = None

    orders_filtered = [
        o
        for o in orders_all
        if order_matches_filters(o, status_filter or None, q, date_from, date_to)
    ]

    # تعرفه‌ها
    tariffs = db.list_tariffs()

    # تیم
    staff = (
        db.list_staff(role="owner", only_active=False)
        + db.list_staff(role="admin", only_active=False)
        + db.list_staff(role="operator", only_active=False)
    )

    # گزارش مالی
    financial = compute_financial_summary(db, orders_all)

    return render_template_string(
        ADMIN_TEMPLATE,
        tab=tab,
        orders=orders_filtered if tab == "orders" else orders_all,
        status_filter=status_filter,
        q=q,
        date_from_str=date_from_str,
        date_to_str=date_to_str,
        tariffs=tariffs,
        staff=staff,
        financial=financial,
    )


# برای Passenger / WSGI
application = app

if __name__ == "__main__":
    app.run(host="0.0.0.0", port=8008, debug=True)
