from __future__ import annotations

import logging
from typing import Any, Dict, List, Optional

from telegram import (
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    Update,
)
from telegram.ext import (
    Application,
    CallbackQueryHandler,
    ContextTypes,
    MessageHandler,
    filters,
)

from .database import Database
from .config import Settings

logger = logging.getLogger("negarprint.staff")


# --------------------------------------------------------------------------- helpers


def _get_db(context: ContextTypes.DEFAULT_TYPE) -> Database:
    db = context.application.bot_data.get("db")
    if not isinstance(db, Database):
        raise RuntimeError("Database not attached to bot_data['db']")
    return db


def is_owner_or_admin(user_id: int, context: ContextTypes.DEFAULT_TYPE) -> bool:
    db = _get_db(context)
    try:
        return db.is_admin(user_id)
    except Exception as exc:  # pragma: no cover
        logger.warning("is_admin failed: %s", exc)
        return False


def ensure_owner(db: Database, settings: Settings) -> None:
    """Make sure at least one owner/admin exists in DB."""
    owner_id = settings.owner_user_id or settings.admin_chat_id
    if not owner_id:
        logger.warning("No OWNER_USER_ID / ADMIN_CHAT_ID configured; skipping ensure_owner")
        return

    try:
        if not db.is_admin(owner_id):
            db.add_staff(owner_id, "owner")
            db.add_staff(owner_id, "admin")
            logger.info("Bootstrap owner/admin with user_id=%s", owner_id)
    except Exception as exc:  # pragma: no cover
        logger.error("ensure_owner failed: %s", exc)


# --------------------------------------------------------------------------- listing UI

PAGE_SIZE = 8


async def show_staff_paged(
    update: Update,
    context: ContextTypes.DEFAULT_TYPE,
    role: str,
    page: int,
) -> None:
    db = _get_db(context)
    rows = db.list_staff(role=role, only_active=False)
    rows = sorted(
        rows,
        key=lambda r: (
            not bool(int(r.get("active", 1) or 1)),
            r.get("created_at"),
            r.get("id"),
        ),
    )

    total = len(rows)
    pages = max(1, (total + PAGE_SIZE - 1) // PAGE_SIZE)
    page = max(0, min(page, pages - 1))
    start = page * PAGE_SIZE
    end = start + PAGE_SIZE
    subset = rows[start:end]

    title = "ادمین‌ها" if role == "admin" else ("اپراتورها" if role == "operator" else role)
    lines: List[str] = [f"👥 لیست {title}: (صفحه {page+1}/{pages})"]

    if not subset:
        lines.append("— خالی —")
    else:
        for r in subset:
            uid = r["user_id"]
            uname = r.get("username") or "-"
            fname = r.get("full_name") or "-"
            duty = "🟢" if bool(int(r.get("on_duty") or 0)) else "⚪️"
            active = "" if bool(int(r.get("active", 1) or 1)) else " (غیرفعال)"
            lines.append(f"{duty} <code>{uid}</code> | {uname} | {fname}{active}")

    kb_rows: List[List[InlineKeyboardButton]] = []

    if pages > 1:
        nav_row: List[InlineKeyboardButton] = []
        if page > 0:
            nav_row.append(
                InlineKeyboardButton("⬅️ قبلی", callback_data=f"STAFF:{role}:PAGE:{page-1}")
            )
        if page < pages - 1:
            nav_row.append(
                InlineKeyboardButton("➡️ بعدی", callback_data=f"STAFF:{role}:PAGE:{page+1}")
            )
        if nav_row:
            kb_rows.append(nav_row)

    kb_rows.append(
        [
            InlineKeyboardButton("➕ افزودن", callback_data=f"STAFF:{role}:ADD"),
            InlineKeyboardButton("🔄 بروزرسانی", callback_data=f"STAFF:{role}:PAGE:{page}"),
        ]
    )

    if subset:
        kb_rows.append(
            [
                InlineKeyboardButton(
                    "❌ حذف با آیدی", callback_data=f"STAFF:{role}:DEL_HELP"
                ),
                InlineKeyboardButton(
                    "🚫 غیرفعال/فعال با آیدی", callback_data=f"STAFF:{role}:TOGGLE_HELP"
                ),
            ]
        )

    markup = InlineKeyboardMarkup(kb_rows)
    text = "\n".join(lines)

    msg = update.effective_message
    try:
        await msg.edit_text(text, parse_mode="HTML", reply_markup=markup)
    except Exception:
        await msg.reply_text(text, parse_mode="HTML", reply_markup=markup)


# --------------------------------------------------------------------------- add / del flow


async def _ask_add_staff(
    update: Update,
    context: ContextTypes.DEFAULT_TYPE,
    role: str,
) -> None:
    context.user_data["staff_add_role"] = role
    context.user_data["staff_add_step"] = "await_input"
    msg = (
        "برای افزودن {role}، یکی از موارد زیر را ارسال کنید:\n"
        "1️⃣ مخاطب (Share Contact)\n"
        "2️⃣ یوزرنیم کاربر به صورت @username\n\n"
        "توجه: اضافه‌کردن صرفاً با شماره موبایل در بات تلگرام پشتیبانی نمی‌شود؛ "
        "کاربر باید حداقل یک‌بار به بات پیام داده یا مخاطب خود را ارسال کند."
    ).format(role=("ادمین" if role == "admin" else "اپراتور"))
    await update.effective_message.reply_text(msg)


async def _handle_add_contact(
    update: Update,
    context: ContextTypes.DEFAULT_TYPE,
) -> None:
    role = context.user_data.get("staff_add_role")
    step = context.user_data.get("staff_add_step")
    if not role or step != "await_input":
        return

    contact = update.effective_message.contact
    if not contact or not contact.user_id:
        return await update.effective_message.reply_text(
            "این مخاطب به حساب کاربری تلگرام متصل نیست. لطفاً مخاطب خود فرد را ارسال کنید."
        )

    db = _get_db(context)
    user_id = int(contact.user_id)
    username = None
    if contact.user_id == update.effective_user.id:
        username = update.effective_user.username
    full_name = f"{contact.first_name or ''} {contact.last_name or ''}".strip() or None

    db.add_staff(user_id, role, username=username, full_name=full_name)
    context.user_data.pop("staff_add_role", None)
    context.user_data.pop("staff_add_step", None)

    await update.effective_message.reply_text(
        f"{'ادمین' if role=='admin' else 'اپراتور'} با موفقیت اضافه شد."
    )


async def _handle_add_text(
    update: Update,
    context: ContextTypes.DEFAULT_TYPE,
) -> None:
    role = context.user_data.get("staff_add_role")
    step = context.user_data.get("staff_add_step")
    if not role or step != "await_input":
        return

    txt = (update.effective_message.text or "").strip()
    if not txt:
        return

    db = _get_db(context)

    if txt.startswith("@"):
        username = txt
        try:
            chat = await context.bot.get_chat(username)
        except Exception:
            return await update.effective_message.reply_text(
                "یوزرنیم پیدا نشد یا بات اجازهٔ دسترسی ندارد."
            )
        user_id = int(chat.id)
        full_name = getattr(chat, "full_name", None)
        if not full_name and getattr(chat, "first_name", None):
            full_name = f"{chat.first_name} {getattr(chat, 'last_name', '')}".strip() or None
        db.add_staff(user_id, role, username=username, full_name=full_name)
        context.user_data.pop("staff_add_role", None)
        context.user_data.pop("staff_add_step", None)
        return await update.effective_message.reply_text(
            f"{'ادمین' if role=='admin' else 'اپراتور'} با موفقیت اضافه شد."
        )

    return await update.effective_message.reply_text(
        "در حال حاضر فقط دو روش پشتیبانی می‌شود:\n"
        "1) ارسال مخاطب (Share Contact)\n"
        "2) ارسال یوزرنیم با @username"
    )


async def staff_on_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
    """Handle callback queries starting with STAFF:..."""
    q = update.callback_query
    if not q:
        return
    data = q.data or ""
    await q.answer()

    parts = data.split(":")
    if len(parts) < 3:
        return
    _, role, cmd, *rest = parts

    if cmd == "PAGE":
        try:
            page = int(rest[0]) if rest else 0
        except Exception:
            page = 0
        return await show_staff_paged(update, context, role, page)

    if cmd == "ADD":
        return await _ask_add_staff(update, context, role)

    if cmd == "DEL_HELP":
        return await q.message.reply_text(
            "برای حذف، آیدی عددی کاربر را به‌صورت:\n"
            f"DEL {role} <user_id>\n"
            "مثال:\nDEL admin 123456789",
        )

    if cmd == "TOGGLE_HELP":
        return await q.message.reply_text(
            "برای فعال/غیرفعال کردن، آیدی عددی کاربر را به‌صورت:\n"
            f"TOGGLE {role} <user_id>",
        )

    return


# --------------------------------------------------------------------------- text commands DEL/TOGGLE


async def handle_staff_text(update: Update, context: ContextTypes.DEFAULT_TYPE) -> bool:
    """Handle textual staff management commands.

    Supported:
      DEL admin 123
      DEL operator 456
      TOGGLE admin 123
    Returns True if the message was handled.
    """
    txt = (update.effective_message.text or "").strip()
    if not txt:
        return False

    parts = txt.split()
    if len(parts) != 3:
        return False

    cmd, role, id_str = parts
    if role not in {"admin", "operator", "owner"}:
        return False
    if cmd.upper() not in {"DEL", "TOGGLE"}:
        return False

    try:
        uid = int(id_str)
    except Exception:
        return False

    db = _get_db(context)

    if cmd.upper() == "DEL":
        n = db.remove_staff(uid, role=role)
        await update.effective_message.reply_text(f"{n} ردیف حذف شد.")
        return True

    if cmd.upper() == "TOGGLE":
        rows = db.list_staff(role=role, only_active=False)
        row = next((r for r in rows if int(r["user_id"]) == uid), None)
        if not row:
            await update.effective_message.reply_text("یافت نشد.")
            return True
        new_active = not bool(int(row.get("active", 1) or 1))
        db.set_active(uid, role, new_active)
        await update.effective_message.reply_text(
            "وضعیت جدید: " + ("فعال ✅" if new_active else "غیرفعال 🚫")
        )
        return True

    return False


# --------------------------------------------------------------------------- registration


def register_staff_handlers(app: Application, db: Database, settings: Settings) -> None:
    """Attach staff-related handlers (contact/text) to the application."""
    app.bot_data["db"] = db
    app.bot_data["settings"] = settings

    async def contact_router(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
        await _handle_add_contact(update, context)

    async def text_router(update: Update, context: ContextTypes.DEFAULT_TYPE) -> None:
        if context.user_data.get("staff_add_role"):
            await _handle_add_text(update, context)
            return
        handled = await handle_staff_text(update, context)
        if handled:
            return

    app.add_handler(MessageHandler(filters.CONTACT, contact_router), group=0)
    app.add_handler(MessageHandler(filters.TEXT & ~filters.COMMAND, text_router), group=1)
