from __future__ import annotations

from collections import defaultdict
from dataclasses import dataclass
from datetime import datetime, timedelta
from typing import Dict, List, Optional, Tuple

from bot.models import Board, Task
from bot.storage import BoardRepository, get_default_repo


@dataclass
class FlowStats:
    total_done: int
    avg_lead_time_days: float
    done_per_day: Dict[str, int]


class TasksService:
    """منطق اصلی کار با تسک‌ها و برد."""

    def __init__(self, repo: BoardRepository | None = None) -> None:
        self.repo: BoardRepository = repo or get_default_repo()

    # ---------- CRUD + حرکت روی برد ----------

    def create_task(self, chat_id: int, title: str, creator_id: int) -> Task:
        board = self.repo.get_board(chat_id)
        task = board.add_task(title, creator_id)
        self.repo.save_board(board)
        return task

    def move_task(self, chat_id: int, task_id: int, new_column_key: str,
                  actor_id: Optional[int]) -> Task:
        board = self.repo.get_board(chat_id)
        task = board.move_task(task_id, new_column_key, actor_id=actor_id)
        self.repo.save_board(board)
        return task

    def block_task(self, chat_id: int, task_id: int, reason: str,
                   actor_id: Optional[int]) -> Task:
        board = self.repo.get_board(chat_id)
        task = board.block_task(task_id, reason, actor_id=actor_id)
        self.repo.save_board(board)
        return task

    def unblock_task(self, chat_id: int, task_id: int,
                     actor_id: Optional[int]) -> Task:
        board = self.repo.get_board(chat_id)
        task = board.unblock_task(task_id, actor_id=actor_id)
        self.repo.save_board(board)
        return task

    def get_board(self, chat_id: int) -> Board:
        return self.repo.get_board(chat_id)

    def assign_task(self, chat_id: int, task_id: int, user_id: int) -> Task:
        board = self.repo.get_board(chat_id)
        if task_id not in board.tasks:
            raise KeyError("Task not found")
        t = board.tasks[task_id]
        t.assignee_id = user_id
        self.repo.save_board(board)
        return t

    # ---------- Queryها ----------

    def list_backlog(self, chat_id: int) -> List[Task]:
        board = self.repo.get_board(chat_id)
        return board.tasks_by_column("backlog")

    def list_tasks_for_member(self, chat_id: int, user_id: int) -> List[Task]:
        board = self.repo.get_board(chat_id)
        return board.tasks_for_member(user_id)

    def list_blocked(self, chat_id: int) -> List[Task]:
        board = self.repo.get_board(chat_id)
        return board.blocked_tasks()

    # ---------- Pull / WIP ----------

    def pull_from_backlog(self, chat_id: int, user_id: int) -> Optional[Task]:
        """یک تسک از Backlog برای کاربر به ستون Ready/Doing می‌کشد (ساده)."""
        board = self.repo.get_board(chat_id)
        backlog = [t for t in board.tasks_by_column("backlog") if not t.is_blocked]
        if not backlog:
            return None
        task = sorted(backlog, key=lambda t: t.id)[0]
        task.assignee_id = user_id
        # سعی می‌کنیم مستقیم به Doing ببریم، اگر WIP پر بود به Ready
        try:
            board.move_task(task.id, "doing", actor_id=user_id)
        except Exception:
            board.move_task(task.id, "ready", actor_id=user_id)
        self.repo.save_board(board)
        return task

    # ---------- ستون‌ها ----------

    def add_column(self, chat_id: int, key: str, name: str, order_index: int,
                   wip_limit: Optional[int] = None, is_done: bool = False) -> None:
        board = self.repo.get_board(chat_id)
        board.add_column(key, name, order_index, wip_limit=wip_limit, is_done_column=is_done)
        self.repo.save_board(board)

    def set_wip_limit(self, chat_id: int, key: str, wip_limit: Optional[int]) -> None:
        board = self.repo.get_board(chat_id)
        board.set_wip_limit(key, wip_limit)
        self.repo.save_board(board)

    # ---------- گزارش جریان (Flow) ----------

    def compute_flow_stats(self, chat_id: int, days: int = 7) -> FlowStats:
        board = self.repo.get_board(chat_id)
        cutoff = datetime.utcnow() - timedelta(days=days)
        done_tasks = [t for t in board.tasks.values() if t.done_at is not None and t.done_at >= cutoff]

        total_done = len(done_tasks)
        if not done_tasks:
            return FlowStats(total_done=0, avg_lead_time_days=0.0, done_per_day={})

        lead_times: List[float] = []
        per_day: Dict[str, int] = defaultdict(int)
        for t in done_tasks:
            lead = (t.done_at - t.created_at).total_seconds() / 86400.0
            lead_times.append(lead)
            day_str = t.done_at.strftime("%Y-%m-%d")
            per_day[day_str] += 1

        avg_lead = sum(lead_times) / len(lead_times)
        return FlowStats(total_done=total_done, avg_lead_time_days=avg_lead, done_per_day=dict(per_day))

    def aging_wip(self, chat_id: int, column_key: str) -> List[Tuple[Task, float]]:
        """تسک‌های درحال‌کار در یک ستون و عمرشان (به روز)."""
        board = self.repo.get_board(chat_id)
        now = datetime.utcnow()
        items: List[Tuple[Task, float]] = []
        for t in board.tasks_by_column(column_key):
            if t.done_at is not None:
                continue
            age_days = (now - t.created_at).total_seconds() / 86400.0
            items.append((t, age_days))
        items.sort(key=lambda x: x[1], reverse=True)
        return items
