from __future__ import annotations

import logging
from typing import Any, Dict, List, Optional

import requests

from config.settings import get_settings

logger = logging.getLogger(__name__)


class BaleAPI:
    """رَپر ساده برای API بله (tapi.bale.ai).

    فقط متدهایی که در این پروژه لازم داریم (getUpdates و sendMessage)
    اینجا پیاده‌سازی شده‌اند.
    """  # noqa: D401

    def __init__(self) -> None:
        self.settings = get_settings()
        base = (self.settings.bale_base_url or "").rstrip("/")
        token = (self.settings.bale_token or "").strip()
        if not token:
            raise RuntimeError("BALE_TOKEN در env تنظیم نشده است.")

        # حالت‌های پشتیبانی‌شده:
        # 1) https://tapi.bale.ai/bot  + token  -> https://tapi.bale.ai/bot<token>
        # 2) https://tapi.bale.ai/bot<token>
        if base.endswith("/bot"):
            self._base_url = f"{base}{token}"
        elif base.endswith(f"/bot{token}"):
            self._base_url = base
        else:
            # اگر کاربر چیز دیگری ست کرده باشد، همان را استفاده می‌کنیم.
            self._base_url = base

        self._session = requests.Session()

    # ------------------------------------------------------------
    # متد داخلی برای ارسال درخواست به API
    # ------------------------------------------------------------
    def _request(self, method: str, method_name: str, **kwargs: Any) -> Dict[str, Any]:
        """ارسال درخواست به متد مشخصی از Bale API.

        برای getUpdates کمی timeout متفاوت در نظر گرفته می‌شود تا
        long-polling بدون ReadTimeout بی‌مورد انجام شود.
        """  # noqa: D401
        url = f"{self._base_url}/{method_name}"

        base_timeout = kwargs.pop("timeout", self.settings.long_poll_timeout)
        connect_timeout = 10
        read_timeout = base_timeout

        if method_name == "getUpdates":
            # کمی حاشیه برای خواندن پاسخ
            read_timeout = base_timeout + 5

        try:
            resp = self._session.request(
                method,
                url,
                timeout=(connect_timeout, read_timeout),
                **kwargs,
            )
            resp.raise_for_status()
        except requests.exceptions.ReadTimeout:
            if method_name == "getUpdates":
                # در long-polling این حالت طبیعی است → فقط result خالی می‌دهیم
                logger.warning(
                    "Bale getUpdates timed out after %s seconds; returning empty list",
                    read_timeout,
                )
                return {"ok": True, "result": []}
            logger.exception("Read timeout while calling Bale API method %s", method_name)
            raise
        except requests.RequestException:
            logger.exception("HTTP error while calling Bale API method %s", method_name)
            raise

        try:
            data: Dict[str, Any] = resp.json()
        except ValueError:
            logger.exception("Invalid JSON response from Bale for method %s", method_name)
            raise

        if not data.get("ok", True) and "result" not in data:
            logger.warning("Bale API error: %s", data)

        return data

    # ------------------------------------------------------------
    # متدهای عمومی موردنیاز پروژه
    # ------------------------------------------------------------
    def get_updates(
        self,
        offset: Optional[int] = None,
        timeout: Optional[int] = None,
    ) -> List[Dict[str, Any]]:
        """خواندن آپدیت‌های بله (long-polling)."""  # noqa: D401
        if timeout is None:
            timeout = self.settings.long_poll_timeout

        params: Dict[str, Any] = {"timeout": timeout}
        if offset is not None:
            params["offset"] = offset

        data = self._request("GET", "getUpdates", params=params, timeout=timeout)
        result = data.get("result", [])
        if not isinstance(result, list):
            logger.warning("Unexpected result type from getUpdates: %r", type(result))
            return []
        return result

    def send_message(
        self,
        chat_id: int,
        text: str,
        reply_to_message_id: Optional[int] = None,
        reply_markup: Optional[Dict[str, Any]] = None,
        parse_mode: Optional[str] = None,
    ) -> Dict[str, Any]:
        """ارسال پیام ساده به یک چت."""  # noqa: D401
        payload: Dict[str, Any] = {
            "chat_id": chat_id,
            "text": text,
        }
        if reply_to_message_id is not None:
            payload["reply_to_message_id"] = reply_to_message_id
        if reply_markup is not None:
            payload["reply_markup"] = reply_markup
        if parse_mode is not None:
            payload["parse_mode"] = parse_mode

        data = self._request("POST", "sendMessage", json=payload, timeout=15)
        return data.get("result", {})
