from __future__ import annotations

import logging
from datetime import datetime
from typing import Any, Dict, List, Optional

from bot.services.bale_api import BaleAPI
from bot.services.tasks_service import TasksService
from bot.services.standup_service import StandupService
from bot.services.report_service import ReportingService
from bot.services.roles_service import (
    RolesService,
    ROLE_OWNER,
    ROLE_PRODUCT_OWNER,
    ROLE_SCRUM_MASTER,
    ROLE_DEVELOPER,
    ROLE_VIEWER,
)
from utils.validators import parse_int, parse_iso_datetime, parse_key_value_args


logger = logging.getLogger(__name__)


class CommandHandler:
    def __init__(
        self,
        api: Optional[BaleAPI] = None,
        tasks_service: Optional[TasksService] = None,
        standup_service: Optional[StandupService] = None,
    ) -> None:
        self.api = api or BaleAPI()
        self.tasks = tasks_service or TasksService()
        self.standup = standup_service or StandupService()
        self.reporting = ReportingService(self.tasks)
        self.roles = RolesService()
        self._pending_new_task_titles: dict[tuple[int, int], bool] = {}

    # -------- entry point --------

    def handle_update(self, update: Dict[str, Any]) -> None:
        # 1) ابتدا callbackهای inline (دکمه‌های داخل پیام) را بررسی می‌کنیم
        callback = update.get("callback_query")
        if callback:
            self._handle_callback(callback)
            return

        # 2) سپس پیام‌های معمولی (متنی / دکمه‌های Reply Keyboard)
        message = update.get("message")
        if not message:
            return

        chat = message.get("chat") or {}
        chat_id = chat.get("id")
        if chat_id is None:
            return

        from_user = message.get("from") or {}
        user_id = from_user.get("id")

        # هر پیام جدید → مطمئن شو این کاربر به‌عنوان عضو در برد ثبت است
        if user_id is not None:
            try:
                member = self.roles.ensure_member_from_update(chat_id, from_user)
                logger.debug("Member ensured: %s", member)
            except Exception:  # noqa: BLE001
                logger.exception("ensure_member_from_update failed")
                member = None
        else:
            member = None

        # اعضای جدیدی که به گروه اضافه شده‌اند (بدون این‌که حتماً پیام بدهند) را هم کش کن
        new_members = message.get("new_chat_members") or []
        for u in new_members:
            try:
                nm = self.roles.ensure_member_from_update(chat_id, u)
                logger.debug("New chat member cached: %s", nm)
            except Exception:  # noqa: BLE001
                logger.exception("ensure_member_from_update for new_chat_members failed")

        # اگر کسی از گروه خارج شد، از کش برد حذفش کن
        left_member = message.get("left_chat_member")
        if left_member:
            lm_id = left_member.get("id")
            if lm_id is not None:
                try:
                    self.roles.remove_member(chat_id, lm_id)
                    logger.debug("Left chat member removed from board: %s", lm_id)
                except Exception:  # noqa: BLE001
                    logger.exception("remove_member failed")

        text: Optional[str] = message.get("text")
        if not text:
            return

        # اگر با / شروع شده → همان رو به عنوان کامند پردازش کن
        if text.startswith("/"):
            parts = text.strip().split(maxsplit=1)
            command = parts[0]
            arg = parts[1] if len(parts) > 1 else ""

            try:
                if command == "/start":
                    self._cmd_show_main_menu(chat_id, user_id, message)
                elif command == "/backlog_new":
                    self._cmd_backlog_new(chat_id, user_id, message, arg)
                elif command == "/backlog":
                    self._cmd_backlog(chat_id, message)
                elif command == "/to_ready":
                    self._cmd_to_ready(chat_id, message, arg)
                elif command == "/board":
                    self._cmd_board(chat_id, message, user_id)
                elif command == "/start_task":
                    self._cmd_move(chat_id, message, user_id, arg, "doing")
                elif command == "/done_task":
                    self._cmd_move(chat_id, message, user_id, arg, "done")
                elif command == "/pull":
                    self._cmd_pull(chat_id, user_id, message)
                elif command == "/mytasks":
                    self._cmd_mytasks(chat_id, user_id, message)
                elif command == "/block":
                    self._cmd_block(chat_id, user_id, message, arg)
                elif command == "/unblock":
                    self._cmd_unblock(chat_id, user_id, message, arg)
                elif command == "/blocked":
                    self._cmd_blocked(chat_id, message)
                elif command == "/board_columns":
                    self._cmd_board_columns(chat_id, message)
                elif command == "/add_column":
                    self._cmd_add_column(chat_id, message, arg)
                elif command == "/set_wip":
                    self._cmd_set_wip(chat_id, message, arg)
                elif command == "/task_edit":
                    self._cmd_task_edit(chat_id, message, arg)
                elif command == "/stats_flow":
                    self._cmd_stats_flow(chat_id, message, arg)
                elif command == "/aging_wip":
                    self._cmd_aging_wip(chat_id, message, arg)
                elif command == "/standup_config":
                    self._cmd_standup_config(chat_id, message, arg)
                elif command == "/standup_answers":
                    self._cmd_standup_answers(chat_id, user_id, message, arg)
                elif command == "/standup_summary":
                    self._cmd_standup_summary(chat_id, message, arg)
                elif command == "/menu":
                    self._cmd_show_main_menu(chat_id, user_id, message)
                else:
                    self.api.send_message(chat_id, "❓ دستور ناشناخته.")
            except Exception as exc:  # noqa: BLE001
                logger.exception("Error while handling %s: %s", command, exc)
                self.api.send_message(chat_id, "❌ خطایی رخ داد. بعداً دوباره تلاش کنید.")
            return

        # اگر کامند نبود، احتمالاً یکی از دکمه‌های منوی کیبورد است
        self._handle_menu_text(chat_id, user_id, message, text)

    # -------- callback (inline keyboard) --------

    def _handle_callback(self, callback: Dict[str, Any]) -> None:
        data = callback.get("data") or ""
        message = callback.get("message") or {}
        chat = message.get("chat") or {}
        chat_id = chat.get("id")
        from_user = callback.get("from") or {}
        user_id = from_user.get("id")

        if chat_id is None or not data:
            return

        # اطمینان از ثبت عضو در برد (برای callbackها)
        if user_id is not None:
            try:
                self.roles.ensure_member_from_update(chat_id, from_user)
            except Exception:  # noqa: BLE001
                logger.exception("ensure_member_from_update failed in callback")

        if data.startswith("task:"):
            parts = data.split(":")
            if len(parts) < 3:
                return
            _, task_id_str, action = parts[:3]
            ok, task_id = parse_int(task_id_str)
            if not ok or task_id is None:
                return
            try:
                self._handle_task_action(chat_id, user_id, message, task_id, action)
            except Exception as exc:  # noqa: BLE001
                logger.exception("Error in task callback: %s", exc)
                self.api.send_message(chat_id, "❌ خطا در انجام عمل روی تسک.")
            return

        if data.startswith("board:"):
            parts = data.split(":")
            if len(parts) >= 3 and parts[1] == "filter":
                mode = parts[2]
                col_key = parts[3] if len(parts) >= 4 else None
                try:
                    self._handle_board_filter(chat_id, message, user_id, mode, col_key)
                except Exception as exc:  # noqa: BLE001
                    logger.exception("Error in board callback: %s", exc)
                    self.api.send_message(chat_id, "❌ خطا در نمایش برد.")
            return

        # مدیریت نقش اعضا
        if data.startswith("member:"):
            # member:<user_id>:roles
            parts = data.split(":")
            if len(parts) >= 3 and parts[2] == "roles":
                target_id_str = parts[1]
                ok, target_id = parse_int(target_id_str)
                if not ok or target_id is None:
                    return
                try:
                    self._show_member_roles_keyboard(chat_id, user_id, message, target_id)
                except Exception as exc:  # noqa: BLE001
                    logger.exception("Error in member roles callback: %s", exc)
                    self.api.send_message(chat_id, "❌ خطا در نمایش نقش‌های کاربر.")
            return

        if data.startswith("setrole:"):
            # setrole:<user_id>:<role>
            parts = data.split(":")
            if len(parts) >= 3:
                target_id_str = parts[1]
                role_key = parts[2]
                ok, target_id = parse_int(target_id_str)
                if not ok or target_id is None:
                    return
                try:
                    self._handle_set_role(chat_id, user_id, message, target_id, role_key)
                except Exception as exc:  # noqa: BLE001
                    logger.exception("Error in setrole callback: %s", exc)
                    self.api.send_message(chat_id, "❌ خطا در تغییر نقش.")
            return

    # -------- task / board actions --------

    def _handle_task_action(self, chat_id: int, user_id: Optional[int],
                            message: Dict[str, Any], task_id: int, action: str) -> None:
        if action == "start":
            self._cmd_move(chat_id, message, user_id, str(task_id), "doing")
        elif action == "ready":
            self._cmd_to_ready(chat_id, message, str(task_id))
        elif action == "done":
            self._cmd_move(chat_id, message, user_id, str(task_id), "done")
        elif action == "block":
            self._cmd_block(chat_id, user_id, message, str(task_id))
        elif action == "unblock":
            self._cmd_unblock(chat_id, user_id, message, str(task_id))

    # -------- helper: keyboards --------

    def _make_backlog_keyboard(self, tasks) -> Dict[str, Any]:
        buttons: List[List[Dict[str, str]]] = []
        for t in tasks:
            buttons.append([{
                "text": f"🔁 به Ready ({t.id})",
                "callback_data": f"task:{t.id}:ready",
            }])
        return {"inline_keyboard": buttons} if buttons else {}

    def _make_mytasks_keyboard(self, tasks) -> Dict[str, Any]:
        buttons: List[List[Dict[str, str]]] = []
        for t in tasks:
            row: List[Dict[str, str]] = []
            if t.column_key in ("backlog", "ready"):
                row.append({
                    "text": f"▶️ Start ({t.id})",
                    "callback_data": f"task:{t.id}:start",
                })
            if t.column_key not in ("done",):
                row.append({
                    "text": f"✅ Done ({t.id})",
                    "callback_data": f"task:{t.id}:done",
                })
            if t.is_blocked:
                row.append({
                    "text": f"♻️ Unblock ({t.id})",
                    "callback_data": f"task:{t.id}:unblock",
                })
            else:
                row.append({
                    "text": f"🚧 Block ({t.id})",
                    "callback_data": f"task:{t.id}:block",
                })
            if row:
                buttons.append(row)
        return {"inline_keyboard": buttons} if buttons else {}

    def _make_blocked_keyboard(self, tasks) -> Dict[str, Any]:
        buttons: List[List[Dict[str, str]]] = []
        for t in tasks:
            buttons.append([{
                "text": f"♻️ Unblock ({t.id})",
                "callback_data": f"task:{t.id}:unblock",
            }])
        return {"inline_keyboard": buttons} if buttons else {}

    def _make_single_task_keyboard(self, task_id: int) -> Dict[str, Any]:
        buttons = [[
            {"text": "▶️ Start", "callback_data": f"task:{task_id}:start"},
            {"text": "✅ Done", "callback_data": f"task:{task_id}:done"},
        ]]
        return {"inline_keyboard": buttons}

    def _make_board_keyboard(self, chat_id: int, user_id: Optional[int]) -> Dict[str, Any]:
        board = self.tasks.get_board(chat_id)
        rows: List[List[Dict[str, str]]] = []

        # ردیف اصلی فیلترها
        row_main: List[Dict[str, str]] = [
            {"text": "📋 همه", "callback_data": "board:filter:all"},
            {"text": "🚧 Blocked", "callback_data": "board:filter:blocked"},
        ]
        if user_id is not None:
            row_main.append({"text": "🙋 فقط من", "callback_data": "board:filter:mine"})
        rows.append(row_main)

        # ستون‌ها
        cols = sorted(board.columns, key=lambda c: c.order_index)
        row: List[Dict[str, str]] = []
        for c in cols:
            row.append({"text": c.name, "callback_data": f"board:filter:col:{c.key}"})
            if len(row) == 3:
                rows.append(row)
                row = []
        if row:
            rows.append(row)

        return {"inline_keyboard": rows} if rows else {}

    # -------- دستورات اصلی تسک/برد --------

    def _cmd_backlog_new(self, chat_id: int, user_id: Optional[int],
                         message: Dict[str, Any], title: str) -> None:
        if not title:
            self.api.send_message(
                chat_id,
                "برای ساخت تسک جدید، عنوان را بعد از /backlog_new بنویس یا از منوی ➕ تعریف کار جدید استفاده کن.",
                reply_to_message_id=message.get("message_id"),
            )
            return
        creator = user_id or 0
        task = self.tasks.create_task(chat_id, title, creator_id=creator)
        self.api.send_message(
            chat_id,
            f"✅ تسک جدید با شناسه {task.id} ساخته شد.",
            reply_to_message_id=message.get("message_id"),
            reply_markup=self._make_single_task_keyboard(task.id),
        )

    def _cmd_backlog(self, chat_id: int, message: Dict[str, Any]) -> None:
        text = self.reporting.format_backlog(chat_id)
        tasks = self.tasks.list_backlog(chat_id)
        kb = self._make_backlog_keyboard(tasks)
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
            reply_markup=kb or None,
        )

    def _cmd_to_ready(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        ok, task_id = parse_int(arg)
        if not ok or task_id is None:
            self.api.send_message(chat_id, "شناسه تسک معتبر نیست.")
            return
        try:
            self.tasks.move_task(chat_id, task_id, "ready", actor_id=None)
            self.api.send_message(chat_id, f"تسک {task_id} به ستون Ready منتقل شد.")
        except Exception as exc:  # noqa: BLE001
            logger.exception("to_ready error: %s", exc)
            self.api.send_message(chat_id, "❌ خطا در انتقال تسک.")

    def _render_board(self, chat_id: int, message: Dict[str, Any],
                       filter_mode: str = "all",
                       column_key: Optional[str] = None,
                       user_id: Optional[int] = None) -> None:
        board = self.tasks.get_board(chat_id)
        lines: List[str] = ["📋 برد کارها:", ""]
        for col in sorted(board.columns, key=lambda c: c.order_index):
            if filter_mode == "col" and column_key and col.key != column_key:
                continue
            lines.append(f"== {col.name} ==")
            for t in sorted(board.tasks_by_column(col.key), key=lambda x: x.id):
                if filter_mode == "mine" and user_id is not None and t.assignee_id != user_id:
                    continue
                if filter_mode == "blocked" and not t.is_blocked:
                    continue
                blocked = "🚧" if t.is_blocked else ""
                assignee = f"(→ {t.assignee_id})" if t.assignee_id else ""
                lines.append(f"{t.id}. {t.title} {blocked} {assignee}")
            lines.append("")
        text = "\n".join(lines)
        kb = self._make_board_keyboard(chat_id, user_id)
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
            reply_markup=kb or None,
        )

    def _cmd_board(self, chat_id: int, message: Dict[str, Any], user_id: Optional[int]) -> None:
        self._render_board(chat_id, message, filter_mode="all", user_id=user_id)

    def _cmd_move(self, chat_id: int, message: Dict[str, Any],
                  user_id: Optional[int], arg: str, new_col: str) -> None:
        ok, task_id = parse_int(arg)
        if not ok or task_id is None:
            self.api.send_message(chat_id, "شناسه تسک معتبر نیست.")
            return
        try:
            t = self.tasks.move_task(chat_id, task_id, new_col, actor_id=user_id)
        except Exception as exc:  # noqa: BLE001
            logger.exception("move error: %s", exc)
            self.api.send_message(chat_id, "❌ خطا در جابجایی تسک.")
            return
        self.api.send_message(
            chat_id,
            f"تسک {t.id} به ستون {new_col} منتقل شد.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_pull(self, chat_id: int, user_id: Optional[int],
                  message: Dict[str, Any]) -> None:
        if user_id is None:
            self.api.send_message(chat_id, "شناسه کاربر نامشخص است.")
            return
        task = self.tasks.pull_from_backlog(chat_id, user_id)
        if not task:
            self.api.send_message(chat_id, "Backlog خالی است یا کاری برای Pull موجود نیست.")
            return
        self.api.send_message(
            chat_id,
            f"🧲 تسک {task.id} برای شما pull شد: {task.title}",
            reply_to_message_id=message.get("message_id"),
            reply_markup=self._make_single_task_keyboard(task.id),
        )

    def _cmd_mytasks(self, chat_id: int, user_id: Optional[int],
                     message: Dict[str, Any]) -> None:
        if user_id is None:
            self.api.send_message(chat_id, "شناسه کاربر نامشخص است.")
            return
        tasks = self.tasks.list_tasks_for_member(chat_id, user_id)
        text = self.reporting.format_tasks_list("🙋 کارهای شما:", tasks)
        kb = self._make_mytasks_keyboard(tasks)
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
            reply_markup=kb or None,
        )

    def _cmd_block(self, chat_id: int, user_id: Optional[int],
                   message: Dict[str, Any], arg: str) -> None:
        ok, task_id = parse_int(arg)
        if not ok or task_id is None:
            self.api.send_message(chat_id, "شناسه تسک معتبر نیست.")
            return
        t = self.tasks.block_task(chat_id, task_id, reason="BLOCKED", actor_id=user_id)
        self.api.send_message(
            chat_id,
            f"🚧 تسک {t.id} بلاک شد.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_unblock(self, chat_id: int, user_id: Optional[int],
                     message: Dict[str, Any], arg: str) -> None:
        ok, task_id = parse_int(arg)
        if not ok or task_id is None:
            self.api.send_message(chat_id, "شناسه تسک معتبر نیست.")
            return
        t = self.tasks.unblock_task(chat_id, task_id, actor_id=user_id)
        self.api.send_message(
            chat_id,
            f"♻️ تسک {t.id} از حالت بلاک خارج شد.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_blocked(self, chat_id: int, message: Dict[str, Any]) -> None:
        tasks = self.tasks.list_blocked(chat_id)
        text = self.reporting.format_tasks_list("🚧 کارهای Blocked:", tasks)
        kb = self._make_blocked_keyboard(tasks)
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
            reply_markup=kb or None,
        )

    def _cmd_board_columns(self, chat_id: int, message: Dict[str, Any]) -> None:
        board = self.tasks.get_board(chat_id)
        lines: List[str] = ["🧱 ستون‌های برد:", ""]
        for c in sorted(board.columns, key=lambda x: x.order_index):
            wip = f"WIP={c.wip_limit}" if c.wip_limit is not None else ""
            done_mark = "✅ Done" if c.is_done_column else ""
            lines.append(f"- {c.key}: {c.name} ({wip} {done_mark})")
        self.api.send_message(
            chat_id,
            "\n".join(lines),
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_add_column(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        args = parse_key_value_args(arg)
        key = args.get("key")
        name = args.get("name", key)
        order_str = args.get("order", "99")
        ok, order = parse_int(order_str)
        if not key or not ok or order is None:
            self.api.send_message(chat_id, "ورودی add_column معتبر نیست. نمونه: key=review name=Review order=5")
            return
        wip_limit = None
        if "wip" in args:
            ok_wip, wip_val = parse_int(args["wip"])
            if ok_wip:
                wip_limit = wip_val
        is_done = args.get("done", "false").lower() in ("1", "true", "yes")
        try:
            self.tasks.add_column(chat_id, key, name, order, wip_limit=wip_limit, is_done=is_done)
            self.api.send_message(chat_id, f"ستون جدید {name} اضافه شد.")
        except Exception as exc:  # noqa: BLE001
            logger.exception("add_column error: %s", exc)
            self.api.send_message(chat_id, "❌ خطا در افزودن ستون.")

    def _cmd_set_wip(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        args = parse_key_value_args(arg)
        key = args.get("key")
        if not key:
            self.api.send_message(chat_id, "نمونه: key=doing wip=3")
            return
        wip = args.get("wip")
        if wip is None:
            limit = None
        else:
            ok, limit = parse_int(wip)
            if not ok:
                self.api.send_message(chat_id, "عدد WIP معتبر نیست.")
                return
        try:
            self.tasks.set_wip_limit(chat_id, key, limit)
            self.api.send_message(chat_id, f"WIP ستون {key} به‌روزرسانی شد.")
        except Exception as exc:  # noqa: BLE001
            logger.exception("set_wip error: %s", exc)
            self.api.send_message(chat_id, "❌ خطا در تنظیم WIP.")

    def _cmd_task_edit(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        self.api.send_message(
            chat_id,
            "ویرایش جزئیات تسک در این نسخه به‌صورت کامل پیاده‌سازی نشده است.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_stats_flow(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        ok, days = parse_int(arg or "7")
        if not ok or days is None:
            days = 7
        stats = self.tasks.compute_flow_stats(chat_id, days=days)
        text = self.reporting.format_flow_stats(stats)
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_aging_wip(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        column_key = arg.strip() or "doing"
        items = self.tasks.aging_wip(chat_id, column_key)
        text = self.reporting.format_aging_wip(items)
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_standup_config(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        args = parse_key_value_args(arg)
        enabled_str = args.get("enabled", "true").lower()
        enabled = enabled_str in ("1", "true", "yes")
        time_str = args.get("time")
        q_raw = args.get("questions")
        time_of_day = None
        if time_str:
            ok, dt = parse_iso_datetime(f"2000-01-01 {time_str}")
            if ok and dt is not None:
                time_of_day = dt.time()
        questions = None
        if q_raw:
            questions = [q.strip() for q in q_raw.split("|") if q.strip()]
        cfg = self.standup.configure(chat_id, enabled=enabled, time_of_day=time_of_day, questions=questions)
        text = f"Standup برای این گروه {'فعال' if cfg.enabled else 'غیرفعال'} شد."
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_standup_answers(self, chat_id: int, user_id: Optional[int],
                             message: Dict[str, Any], arg: str) -> None:
        if user_id is None:
            self.api.send_message(chat_id, "شناسه کاربر نامشخص است.")
            return
        answers = [a.strip() for a in arg.split("|") if a.strip()]
        if not answers:
            self.api.send_message(chat_id, "پاسخ‌ها را با | از هم جدا کن.")
            return
        self.standup.add_answers(chat_id, user_id, answers)
        self.api.send_message(
            chat_id,
            "✅ پاسخ‌های Standup ثبت شد.",
            reply_to_message_id=message.get("message_id"),
        )

    def _cmd_standup_summary(self, chat_id: int, message: Dict[str, Any], arg: str) -> None:
        ok, dt = parse_iso_datetime(arg or datetime.utcnow().strftime("%Y-%m-%d"))
        if not ok or dt is None:
            self.api.send_message(chat_id, "تاریخ معتبر نیست (فرمت: YYYY-MM-DD).")
            return
        entries = self.standup.entries_for_day(chat_id, dt.date())
        if not entries:
            self.api.send_message(chat_id, "برای این روز هیچ Standupای ثبت نشده است.")
            return
        lines: List[str] = [f"📣 Standup {dt.date()}:", ""]
        for e in entries:
            lines.append(f"- کاربر {e.user_id}:")
            for ans in e.answers:
                lines.append(f"  • {ans}")
            lines.append("")
        self.api.send_message(
            chat_id,
            "\n".join(lines),
            reply_to_message_id=message.get("message_id"),
        )

    # -------- Board filter via inline callback --------

    def _handle_board_filter(self, chat_id: int, message: Dict[str, Any],
                             user_id: Optional[int], mode: str,
                             column_key: Optional[str]) -> None:
        if mode == "all":
            self._render_board(chat_id, message, filter_mode="all", user_id=user_id)
        elif mode == "blocked":
            self._render_board(chat_id, message, filter_mode="blocked", user_id=user_id)
        elif mode == "col" and column_key:
            self._render_board(chat_id, message, filter_mode="col", column_key=column_key, user_id=user_id)
        elif mode == "mine":
            self._render_board(chat_id, message, filter_mode="mine", user_id=user_id)

    # -------- منوی اصلی و نقش‌ها --------

    def _build_main_menu_keyboard(self, chat_id: int, user_id: Optional[int]) -> Dict[str, Any]:
        if user_id is None:
            keyboard = [
                [{"text": "📋 برد"}],
            ]
            return {"keyboard": keyboard, "resize_keyboard": True}

        ok, role = self.roles.member_has_any_role(
            chat_id, user_id,
            [ROLE_PRODUCT_OWNER, ROLE_SCRUM_MASTER, ROLE_DEVELOPER, ROLE_VIEWER],
        )
        role = role or ROLE_VIEWER

        keyboard: List[List[Dict[str, str]]] = []

        keyboard.append([
            {"text": "📋 برد"},
            {"text": "🙋 کارهای من"},
        ])
        keyboard.append([
            {"text": "🧲 Pull کار"},
            {"text": "🚧 کارهای Blocked"},
        ])
        keyboard.append([
            {"text": "📣 Standup امروز"},
        ])

        if role in (ROLE_PRODUCT_OWNER, ROLE_SCRUM_MASTER):
            keyboard.append([
                {"text": "➕ تعریف کار جدید"},
            ])

        has_owner, _ = self.roles.member_has_any_role(chat_id, user_id, [ROLE_OWNER])
        if has_owner:
            keyboard.append([
                {"text": "🧱 ستون‌ها / WIP"},
                {"text": "👥 اعضا / نقش‌ها"},
            ])
            keyboard.append([
                {"text": "📊 آمار جریان"},
            ])

        return {"keyboard": keyboard, "resize_keyboard": True}

    def _cmd_show_main_menu(self, chat_id: int, user_id: Optional[int], message: Dict[str, Any]) -> None:
        kb = self._build_main_menu_keyboard(chat_id, user_id)
        self.api.send_message(
            chat_id,
            "📋 منوی اصلی ربات مدیریت کار (براساس نقش شما):",
            reply_to_message_id=message.get("message_id"),
            reply_markup=kb,
        )

    def _handle_menu_text(self, chat_id: int, user_id: Optional[int],
                          message: Dict[str, Any], text: str) -> None:
        """دکمه‌های Reply Keyboard و حالت‌های ساده‌ی منو را هندل می‌کند."""
        if user_id is None:
            return

        key = (chat_id, user_id)
        if self._pending_new_task_titles.get(key):
            if text.startswith("/"):
                self._pending_new_task_titles.pop(key, None)
                return
            self._cmd_backlog_new(chat_id, user_id, message, text)
            self._pending_new_task_titles.pop(key, None)
            self._cmd_show_main_menu(chat_id, user_id, message)
            return

        label = text.strip()

        if label.startswith("📋 برد"):
            self._cmd_board(chat_id, message, user_id)
            return
        if label.startswith("🙋 کارهای من"):
            self._cmd_mytasks(chat_id, user_id, message)
            return
        if label.startswith("🧲 Pull کار"):
            self._cmd_pull(chat_id, user_id, message)
            return
        if label.startswith("🚧 کارهای Blocked"):
            self._cmd_blocked(chat_id, message)
            return
        if label.startswith("📣 Standup"):
            today_str = datetime.utcnow().strftime("%Y-%m-%d")
            self._cmd_standup_summary(chat_id, message, today_str)
            return
        if label.startswith("➕ تعریف کار جدید"):
            allowed = [ROLE_PRODUCT_OWNER]
            has_perm, _ = self.roles.member_has_any_role(chat_id, user_id, allowed)
            owner_ok, _ = self.roles.member_has_any_role(chat_id, user_id, [ROLE_OWNER])
            if not (has_perm or owner_ok):
                self.api.send_message(
                    chat_id,
                    "فقط مدیر محصول (یا مالک برد) می‌تواند کار جدید تعریف کند.",
                    reply_to_message_id=message.get("message_id"),
                )
                return
            self._pending_new_task_titles[key] = True
            self.api.send_message(
                chat_id,
                "📝 عنوان تسک جدید را بفرست (فقط متن عنوان).",
                reply_to_message_id=message.get("message_id"),
            )
            return
        if label.startswith("🧱 ستون‌ها / WIP"):
            owner_ok, _ = self.roles.member_has_any_role(chat_id, user_id, [ROLE_OWNER])
            if not owner_ok:
                self.api.send_message(
                    chat_id,
                    "فقط مالک برد می‌تواند ستون‌ها و WIP را مدیریت کند.",
                    reply_to_message_id=message.get("message_id"),
                )
                return
            self._cmd_board_columns(chat_id, message)
            return
        if label.startswith("📊 آمار جریان"):
            owner_ok, _ = self.roles.member_has_any_role(chat_id, user_id, [ROLE_OWNER, ROLE_SCRUM_MASTER])
            if not owner_ok:
                self.api.send_message(
                    chat_id,
                    "فقط Scrum Master یا مالک برد می‌توانند آمار جریان را ببینند.",
                    reply_to_message_id=message.get("message_id"),
                )
                return
            arg = "7"
            self._cmd_stats_flow(chat_id, message, arg)
            return
        if label.startswith("👥 اعضا / نقش‌ها"):
            self._cmd_members_overview(chat_id, user_id, message)
            return

    def _cmd_members_overview(self, chat_id: int, user_id: Optional[int], message: Dict[str, Any]) -> None:
        if user_id is None:
            return
        owner_ok, _ = self.roles.member_has_any_role(chat_id, user_id, [ROLE_OWNER])
        if not owner_ok:
            self.api.send_message(
                chat_id,
                "فقط مالک برد می‌تواند اعضا و نقش‌ها را مدیریت کند.",
                reply_to_message_id=message.get("message_id"),
            )
            return

        members = self.roles.list_members(chat_id)
        if not members:
            self.api.send_message(
                chat_id,
                "هنوز هیچ عضوی در این برد ثبت نشده است (ربات هنوز پیامی از کسی ندیده).",
                reply_to_message_id=message.get("message_id"),
            )
            return

        lines: List[str] = ["👥 اعضای شناخته‌شده‌ی این گروه:", ""]
        keyboard: List[List[Dict[str, str]]] = []
        for m in members:
            role_label = {
                ROLE_OWNER: "Owner",
                ROLE_PRODUCT_OWNER: "Product Owner",
                ROLE_SCRUM_MASTER: "Scrum Master",
                ROLE_DEVELOPER: "Developer",
                ROLE_VIEWER: "Viewer",
            }.get(m.role, m.role)
            owner_mark = " 👑" if m.is_owner else ""
            lines.append(f"- {m.display_name}{owner_mark} → {role_label}")
            keyboard.append([{
                "text": f"تغییر نقش {m.display_name}",
                "callback_data": f"member:{m.user_id}:roles",
            }])

        text = "\n".join(lines)
        reply_markup = {"inline_keyboard": keyboard}
        self.api.send_message(
            chat_id,
            text,
            reply_to_message_id=message.get("message_id"),
            reply_markup=reply_markup,
        )

    def _show_member_roles_keyboard(self, chat_id: int, user_id: Optional[int],
                                    message: Dict[str, Any], target_user_id: int) -> None:
        if user_id is None:
            return
        owner_ok, _ = self.roles.member_has_any_role(chat_id, user_id, [ROLE_OWNER])
        if not owner_ok:
            self.api.send_message(
                chat_id,
                "شما اجازه‌ی تغییر نقش را ندارید.",
            )
            return

        members = self.roles.list_members(chat_id)
        target = next((m for m in members if m.user_id == target_user_id), None)
        if not target:
            self.api.send_message(chat_id, "کاربر در لیست اعضا یافت نشد.")
            return

        if target.is_owner:
            self.api.send_message(chat_id, "نقش owner قابل تغییر نیست.")
            return

        text = f"نقش جدید برای {target.display_name} را انتخاب کن:"
        keyboard: List[List[Dict[str, str]]] = []

        role_options = [
            (ROLE_PRODUCT_OWNER, "Product Owner"),
            (ROLE_SCRUM_MASTER, "Scrum Master"),
            (ROLE_DEVELOPER, "Developer"),
            (ROLE_VIEWER, "Viewer"),
        ]
        for role_key, caption in role_options:
            keyboard.append([{
                "text": caption,
                "callback_data": f"setrole:{target.user_id}:{role_key}",
            }])

        reply_markup = {"inline_keyboard": keyboard}
        self.api.send_message(chat_id, text, reply_markup=reply_markup)

    def _handle_set_role(self, chat_id: int, user_id: Optional[int],
                         message: Dict[str, Any], target_user_id: int, role_key: str) -> None:
        if user_id is None:
            return
        try:
            updated = self.roles.set_role(chat_id, user_id, target_user_id, role_key)
        except PermissionError as exc:
            self.api.send_message(chat_id, f"⛔ {exc}")
            return
        except Exception as exc:  # noqa: BLE001
            logger.exception("set_role error: %s", exc)
            self.api.send_message(chat_id, "❌ خطا در تغییر نقش.")
            return

        self.api.send_message(
            chat_id,
            f"✅ نقش جدید برای {updated.display_name} تنظیم شد: {updated.role}",
        )
