from __future__ import annotations

from typing import List

from bot.models import Board, Task
from bot.storage.memory_repo import InMemoryBoardRepository


class TasksService:
    """Application service for Scrumban board & tasks (MVP + /pull)."""

    def __init__(self, repo: InMemoryBoardRepository | None = None) -> None:
        self.repo = repo or InMemoryBoardRepository()

    def get_board(self, chat_id: int) -> Board:
        return self.repo.get_board(chat_id)

    def create_task(self, chat_id: int, title: str, creator_id: int) -> Task:
        board = self.get_board(chat_id)
        return board.add_task(title=title, creator_id=creator_id)

    def move_task(self, chat_id: int, task_id: int, new_column_key: str) -> Task:
        board = self.get_board(chat_id)
        return board.move_task(task_id, new_column_key)

    def list_tasks_by_column(self, chat_id: int, column_key: str) -> List[Task]:
        board = self.get_board(chat_id)
        return board.tasks_by_column(column_key)

    def list_tasks_for_member(self, chat_id: int, member_id: int) -> List[Task]:
        board = self.get_board(chat_id)
        return board.tasks_for_member(member_id)

    def pull_task_for_member(self, chat_id: int, member_id: int) -> Task:
        """Simple Pull system: pick the lowest-id Ready task and move to Doing."""
        board = self.get_board(chat_id)
        ready_tasks = sorted(
            (t for t in board.tasks.values() if t.column_key == "ready"),
            key=lambda t: t.id,
        )
        if not ready_tasks:
            raise LookupError("هیچ کاری در ستون Ready وجود ندارد.")

        task = ready_tasks[0]
        task.assignee_id = member_id
        board.move_task(task.id, "doing")
        return task
