from __future__ import annotations
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from sqlalchemy import select
from app.db.session import get_db
from app.api.dependencies import get_tenant_context
from app.models import Tenant, StoreProfile, AppUser, StoreCustomer
from app.schemas.tenant import StoreChannelInfo
from app.schemas.customer import CustomerSelfRegister, CustomerOut
from app.core.security import hash_password
from app.utils.validators import ensure_iran_phone


router = APIRouter(prefix="/public", tags=["public"])


@router.get("/tenants/{tenant_id}/channel", response_model=StoreChannelInfo)
def tenant_channel_info(tenant_id: str, db: Session = Depends(get_db)) -> StoreChannelInfo:
    tenant = db.get(Tenant, tenant_id)
    if not tenant or tenant.status != "active":
        raise HTTPException(status_code=404, detail="tenant not found")
    profile = db.get(StoreProfile, tenant_id)
    return StoreChannelInfo(
        tenant_id=tenant.id,
        channel_id=profile.channel_id if profile else None,
        channel_invite_link=profile.channel_invite_link if profile else None,
    )


@router.post("/customers/register", response_model=CustomerOut)
def customer_self_register(
    payload: CustomerSelfRegister,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    ensure_iran_phone(payload.phone)
    existing = db.scalars(
        select(AppUser).where(AppUser.tenant_id == tenant.id, AppUser.login_username == payload.phone)
    ).first()
    if existing:
        raise HTTPException(status_code=400, detail="account already exists")
    user = AppUser(
        tenant_id=tenant.id,
        login_username=payload.phone,
        password_hash=hash_password(payload.password),
        role="customer",
        phone=payload.phone,
        status="active",
    )
    db.add(user)
    db.flush()
    customer = StoreCustomer(
        tenant_id=tenant.id,
        user_id=user.id,
        first_name=payload.first_name or "مشتری",
        last_name=payload.last_name or "جدید",
        phone=payload.phone,
        postal_code=payload.postal_code,
        address=payload.address,
        delivery_geo=(
            {"lat": payload.delivery_lat, "lng": payload.delivery_lng}
            if payload.delivery_lat is not None and payload.delivery_lng is not None
            else None
        ),
    )
    db.add(customer)
    db.commit()
    db.refresh(customer)
    return CustomerOut(
        id=customer.id,
        user_id=customer.user_id,
        first_name=customer.first_name,
        last_name=customer.last_name,
        phone=customer.phone,
        postal_code=customer.postal_code,
        address=customer.address,
    )
