from __future__ import annotations

import json
from typing import Any
from uuid import UUID

import httpx
from fastapi import APIRouter, Depends, HTTPException, Request
from pydantic import BaseModel
from sqlalchemy.orm import Session

from app.api.dependencies import get_db
from app.core.config import get_settings
from app.core.logging import get_logger
from app.models import AccountRequest
from app.schemas.account_purchase import AccountPurchaseInit, AccountPurchaseInitResponse
from app.schemas.tenant import TenantCreate
from app.services.tenants import create_tenant


router = APIRouter(prefix="/public/account-purchase", tags=["public"])

logger = get_logger(__name__)
settings = get_settings()


def _zarinpal_base_url() -> str:
    sandbox = getattr(settings, "ZARINPAL_SANDBOX", False)
    return "https://sandbox.zarinpal.com/pg" if sandbox else "https://api.zarinpal.com/pg"


def _zarinpal_startpay_url(authority: str) -> str:
    sandbox = getattr(settings, "ZARINPAL_SANDBOX", False)
    host = "sandbox.zarinpal.com" if sandbox else "www.zarinpal.com"
    return f"https://{host}/pg/StartPay/{authority}"


@router.post("/init", response_model=AccountPurchaseInitResponse)
def account_purchase_init(
    payload: AccountPurchaseInit,
    db: Session = Depends(get_db),
) -> AccountPurchaseInitResponse:
    merchant_id = getattr(settings, "ZARINPAL_MERCHANT_ID", None)
    amount = getattr(settings, "ACCOUNT_PURCHASE_AMOUNT", None)
    callback_url = getattr(settings, "ZARINPAL_CALLBACK_URL", None)
    if not merchant_id or not amount or not callback_url:
        raise HTTPException(status_code=500, detail="payment gateway not configured")

    extra = {
        "store_title": payload.store_title,
        "owner_full_name": payload.owner_full_name,
        "address": payload.address,
        "business_description": payload.business_description,
        "lat": payload.lat,
        "lng": payload.lng,
        "bot_token": payload.bot_token,
        "channel_id": payload.channel_id,
        "channel_link": payload.channel_link,
        "telegram_user_id": payload.telegram_user_id,
    }
    lead = AccountRequest(
        full_name=payload.owner_full_name,
        phone=payload.phone,
        description=json.dumps(extra, ensure_ascii=False),
    )
    db.add(lead)
    db.commit()
    db.refresh(lead)

    base = _zarinpal_base_url()
    amount_value = int(amount)
    req_body = {
        "merchant_id": merchant_id,
        "amount": amount_value,
        "callback_url": callback_url,
        "description": f"خرید حساب فروشگاه برای {payload.store_title}",
        "metadata": {"mobile": payload.phone},
    }
    try:
        with httpx.Client(timeout=10) as client:
            resp = client.post(f"{base}/v4/payment/request.json", json=req_body)
            resp.raise_for_status()
    except httpx.HTTPError as exc:
        logger.exception("zarinpal payment request failed", exc_info=exc)
        raise HTTPException(status_code=502, detail="payment gateway error") from exc

    data = resp.json().get("data") or {}
    authority = data.get("authority")
    if not authority or data.get("code") != 100:
        logger.error("zarinpal request error response", extra={"response": resp.text})
        raise HTTPException(status_code=502, detail="payment gateway rejected request")

    payment_url = _zarinpal_startpay_url(authority)
    return AccountPurchaseInitResponse(payment_url=payment_url, request_id=str(lead.id))


class AttachPaymentMessage(BaseModel):
    request_id: str
    payment_message_id: int


@router.post("/attach-payment-message", response_model=dict)
def attach_payment_message(
    payload: AttachPaymentMessage,
    db: Session = Depends(get_db),
) -> dict:
    """
    ذخیره آیدی پیام پرداخت در رکورد درخواست حساب، تا بعد از نهایی شدن پرداخت
    بتوانیم پیام قبلی را در تلگرام ویرایش کنیم (حذف دکمه و تغییر متن).
    """
    try:
        lead_id = UUID(payload.request_id)
    except Exception as exc:
        raise HTTPException(status_code=400, detail="invalid request id") from exc

    lead = db.get(AccountRequest, lead_id)
    if not lead:
        raise HTTPException(status_code=404, detail="request not found")

    try:
        extra = json.loads(lead.description or "{}")
    except json.JSONDecodeError:
        extra = {}

    extra["payment_message_id"] = payload.payment_message_id
    lead.description = json.dumps(extra, ensure_ascii=False)
    db.add(lead)
    db.commit()

    return {"ok": True}


@router.get("/callback", response_model=dict)
def account_purchase_callback(
    request: Request,
    db: Session = Depends(get_db),
) -> dict:
    merchant_id = getattr(settings, "ZARINPAL_MERCHANT_ID", None)
    amount = getattr(settings, "ACCOUNT_PURCHASE_AMOUNT", None)
    if not merchant_id or not amount:
        raise HTTPException(status_code=500, detail="payment gateway not configured")

    params = request.query_params
    authority = params.get("Authority") or params.get("authority")
    status_param = params.get("Status") or params.get("status")
    req_id = params.get("req") or params.get("request_id")
    if not authority or not status_param or not req_id:
        raise HTTPException(status_code=400, detail="invalid callback parameters")

    try:
        lead_id = UUID(req_id)
    except Exception as exc:
        raise HTTPException(status_code=400, detail="invalid request id") from exc

    lead = db.get(AccountRequest, lead_id)
    if not lead:
        raise HTTPException(status_code=404, detail="request not found")

    if status_param.lower() != "ok":
        lead.status = "rejected"
        db.add(lead)
        db.commit()
        return {"ok": False, "reason": "payment_not_ok"}

    base = _zarinpal_base_url()
    amount_value = int(amount)
    verify_body = {
        "merchant_id": merchant_id,
        "amount": amount_value,
        "authority": authority,
    }
    try:
        with httpx.Client(timeout=10) as client:
            resp = client.post(f"{base}/v4/payment/verify.json", json=verify_body)
            resp.raise_for_status()
    except httpx.HTTPError as exc:
        logger.exception("zarinpal payment verify failed", exc_info=exc)
        raise HTTPException(status_code=502, detail="payment gateway error") from exc

    data = resp.json().get("data") or {}
    if data.get("code") != 100:
        lead.status = "failed"
        db.add(lead)
        db.commit()
        return {"ok": False, "reason": "payment_not_verified"}

    if lead.status == "done":
        return {"ok": True, "already": True}

    try:
        extra = json.loads(lead.description or "{}")
    except json.JSONDecodeError:
        extra = {}

    geo = None
    if extra.get("lat") is not None and extra.get("lng") is not None:
        geo = {"lat": extra.get("lat"), "lng": extra.get("lng")}

    tenant_payload = TenantCreate(
        code=None,
        title=extra.get("store_title") or lead.full_name or "Store",
        category=None,
        address=extra.get("address"),
        geo_location=geo,
        work_hours=None,
        subscription_tier="base",
        store_admin_phone=lead.phone or "",
        store_admin_username=None,
        store_admin_password=None,
        bot_token=extra.get("bot_token") or "",
        contact_phone=lead.phone or None,
    )

    tenant, username, password = create_tenant(db, tenant_payload)  # type: ignore[arg-type]

    channel_payload: dict[str, Any] = {}
    if extra.get("channel_id"):
        channel_payload["channel_id"] = extra["channel_id"]
    if extra.get("channel_link"):
        channel_payload["channel_invite_link"] = extra["channel_link"]
    if channel_payload:
        from app.api.routers.tenants import update_profile  # lazy import to avoid cycle
        from app.schemas.tenant import StoreProfileIn

        tenant_id_str = str(tenant.id)
        profile_in = StoreProfileIn(**channel_payload)
        update_profile(tenant_id_str, profile_in, db)

    lead.status = "done"
    db.add(lead)
    db.commit()

    # ارسال پیام تلگرام برای مدیر
    try:
        telegram_id = int(extra.get("telegram_user_id"))
    except Exception:
        telegram_id = None

    # در صورت موجود بودن، آیدی پیام پرداخت (برای حذف دکمه و تغییر متن)
    payment_message_id = extra.get("payment_message_id")

    if telegram_id and getattr(settings, "HYPER_BOT_TOKEN", None):
        telegram_base = getattr(settings, "TELEGRAM_API_BASE", "https://api.telegram.org")
        bot_base = f"{telegram_base}/bot{settings.HYPER_BOT_TOKEN}"
        try:
            with httpx.Client(timeout=10) as client:
                # اگر آیدی پیام پرداخت را داریم، ابتدا دکمه را حذف و متن را به وضعیت موفقیت تغییر می‌دهیم
                if payment_message_id:
                    try:
                        client.post(
                            f"{bot_base}/editMessageReplyMarkup",
                            json={
                                "chat_id": telegram_id,
                                "message_id": payment_message_id,
                                "reply_markup": None,
                            },
                        )
                        client.post(
                            f"{bot_base}/editMessageText",
                            json={
                                "chat_id": telegram_id,
                                "message_id": payment_message_id,
                                "text": "✅ پرداخت شما با موفقیت تأیید شد.\n"
                                        "در حال آماده‌سازی و ارسال اطلاعات فروشگاه هستیم...",
                            },
                        )
                    except Exception:
                        # اگر ویرایش پیام پرداخت با خطا روبه‌رو شد، ادامه می‌دهیم تا حداقل پیام نهایی ارسال شود
                        pass

                # ارسال پیام نهایی با اطلاعات فروشگاه
                text = (
                    "حساب فروشگاه شما با موفقیت ایجاد شد ✅\n\n"
                    f"کد فروشگاه: {getattr(tenant, 'code', '')}\n"
                    f"نام کاربری پنل مدیر: {username}\n"
                    f"رمز عبور پنل مدیر: {password}\n"
                    f"شناسه فروشگاه: {tenant.id}\n"
                )
                client.post(
                    f"{bot_base}/sendMessage",
                    json={"chat_id": telegram_id, "text": text},
                )
        except Exception as exc:
            logger.exception("failed to send telegram notification", exc_info=exc)

    return {"ok": True, "tenant_id": str(tenant.id)}
