from __future__ import annotations
from typing import Any
from aiogram import Router, F, types
from aiogram.filters import CommandStart, Command
from aiogram.fsm.context import FSMContext
from app.bots.hyper.states import AdminStates, RegisterMarketStates, ManageTenantStates
from app.bots.hyper import keyboards
from app.bots.shared.api_client import BackendClient
from app.core.config import get_settings
from app.core.logging import get_logger


router = Router()
logger = get_logger("app.bots.hyper")
settings = get_settings()


def _parse_allowed_admins(raw: str | None) -> set[int]:
    if not raw:
        return set()
    result: set[int] = set()
    for chunk in raw.split(","):
        chunk = chunk.strip()
        if not chunk:
            continue
        try:
            result.add(int(chunk))
        except ValueError:
            continue
    return result


ALLOWED_ADMIN_IDS = _parse_allowed_admins(settings.HYPER_ADMIN_TELEGRAM_IDS)

REGISTER_FLOW = [
    {"key": "title", "label": "نام مارکت", "type": "text"},
    {"key": "store_admin_phone", "label": "موبایل مدیر مارکت", "type": "text"},
    {"key": "address", "label": "آدرس مارکت", "type": "text"},
    {"key": "category", "label": "دسته‌بندی صنفی", "type": "text"},
    {"key": "bot_token", "label": "توکن ربات فروشگاه", "type": "text"},
    {"key": "channel_id", "label": "شناسه کانال فروشگاه", "type": "text"},
    {"key": "channel_invite_link", "label": "لینک دعوت کانال", "type": "text"},
    {"key": "work_hours", "label": "ساعات پاسخ‌گویی", "type": "text"},
    {"key": "subscription_tier", "label": "نوع اشتراک", "type": "choice"},
]
REGISTER_MAP = {step["key"]: step for step in REGISTER_FLOW}


# ---------------------------------------------------------------------------
# Helpers
# ---------------------------------------------------------------------------


async def _get_client(state: FSMContext) -> BackendClient:
    data = await state.get_data()
    token = data.get("token")
    if not token:
        raise RuntimeError("missing token")
    return BackendClient(token=token)


async def _reset_register_flow(state: FSMContext) -> None:
    await state.update_data(register_index=0, register_data={}, register_waiting=None)


async def _current_step(state: FSMContext) -> dict[str, Any] | None:
    data = await state.get_data()
    idx = data.get("register_index", 0)
    if idx >= len(REGISTER_FLOW):
        return None
    return REGISTER_FLOW[idx]


async def _show_register_prompt(target: types.Message | types.CallbackQuery, state: FSMContext) -> None:
    step = await _current_step(state)
    if step is None:
        await _show_register_summary(target, state)
        return
    data = await state.get_data()
    register_data: dict[str, Any] = data.get("register_data", {})
    idx = data.get("register_index", 0)
    text = f"مرحله {idx + 1}/{len(REGISTER_FLOW)} – {step['label']}"
    current_value = register_data.get(step["key"])
    if current_value:
        text += f"\nمقدار فعلی: {current_value}"

    if isinstance(target, types.CallbackQuery):
        message = target.message
    else:
        message = target

    if step["type"] == "choice":
        await message.answer(text, reply_markup=keyboards.register_tier_keyboard())
    else:
        await message.answer(text, reply_markup=keyboards.register_input_keyboard(step["key"]))


async def _show_register_summary(target: types.Message | types.CallbackQuery, state: FSMContext) -> None:
    data = await state.get_data()
    register_data: dict[str, Any] = data.get("register_data", {})
    lines = [f"{REGISTER_MAP[key]['label']}: {value}" for key, value in register_data.items()]
    summary = "پیش‌نمایش مارکت:\n" + "\n".join(lines)
    if isinstance(target, types.CallbackQuery):
        await target.message.answer(summary, reply_markup=keyboards.register_confirm_keyboard())
    else:
        await target.answer(summary, reply_markup=keyboards.register_confirm_keyboard())


async def _cancel_register_flow(message: types.Message | types.CallbackQuery, state: FSMContext) -> None:
    await _reset_register_flow(state)
    await state.set_state(AdminStates.menu)
    text = "عملیات ثبت مارکت لغو شد."
    if isinstance(message, types.CallbackQuery):
        await message.answer()
        await message.message.answer(text, reply_markup=keyboards.hyper_admin_menu())
    else:
        await message.answer(text, reply_markup=keyboards.hyper_admin_menu())


# ---------------------------------------------------------------------------
# Entry points
# ---------------------------------------------------------------------------


@router.message(CommandStart())
async def start(message: types.Message, state: FSMContext):
    await state.clear()
    await message.answer(
        "به ربات مدیریتی هایپر خوش آمدید. برای ادامه، یکی از گزینه‌های زیر را انتخاب کنید.",
        reply_markup=keyboards.main_menu(),
    )


@router.message(F.text == "🔐 ورود مدیر هایپر")
async def hyper_login(message: types.Message, state: FSMContext):
    if ALLOWED_ADMIN_IDS and message.from_user.id not in ALLOWED_ADMIN_IDS:
        await message.answer("دسترسی شما برای مدیریت هایپر مجاز نیست.")
        return
    waiting = await message.answer("⌛ در حال ورود...")
    client = BackendClient()
    try:
        auth = await client.login(settings.DEFAULT_HYPER_ADMIN_USERNAME, settings.DEFAULT_HYPER_ADMIN_PASSWORD)
    except Exception as exc:
        await waiting.edit_text(f"ورود ناموفق بود: {exc}")
        logger.exception("hyper admin login failed")
        return
    await state.update_data(token=auth["token"])
    await state.set_state(AdminStates.menu)
    await waiting.edit_text("ورود موفقیت‌آمیز بود.")
    await message.answer("از منوی زیر گزینه‌ای را انتخاب کنید.", reply_markup=keyboards.hyper_admin_menu())


@router.message(AdminStates.menu, F.text == "↩️ خروج")
async def hyper_logout(message: types.Message, state: FSMContext):
    await state.clear()
    await message.answer("از حساب خارج شدید.", reply_markup=keyboards.main_menu())


# ---------------------------------------------------------------------------
# Register market flow
# ---------------------------------------------------------------------------


@router.message(AdminStates.menu, F.text == "➕ ثبت مارکت جدید")
async def register_market_entry(message: types.Message, state: FSMContext):
    await _reset_register_flow(state)
    await state.set_state(RegisterMarketStates.flow)
    await message.answer("ویزارد ثبت مارکت شروع شد. برای هر مرحله از دکمه‌ها استفاده کنید.")
    await _show_register_prompt(message, state)


@router.callback_query(RegisterMarketStates.flow, F.data.startswith("reg:input:"))
async def register_request_input(callback: types.CallbackQuery, state: FSMContext):
    _, _, step_key = callback.data.partition("reg:input:")
    step = REGISTER_MAP.get(step_key)
    if not step:
        await callback.answer("مرحله نامعتبر است.", show_alert=True)
        return
    await state.set_state(RegisterMarketStates.awaiting_value)
    await state.update_data(register_waiting=step_key)
    await callback.answer()
    await callback.message.answer(f"لطفاً {step['label']} را ارسال کنید. برای لغو، /cancel را بزنید.")


@router.message(RegisterMarketStates.awaiting_value)
async def register_receive_value(message: types.Message, state: FSMContext):
    data = await state.get_data()
    step_key = data.get("register_waiting")
    if not step_key:
        await message.answer("مرحله‌ای در انتظار مقدار نیست.")
        return
    value = message.text.strip()
    if not value:
        await message.answer("ورودی نمی‌تواند خالی باشد.")
        return
    register_data = data.get("register_data", {})
    register_data[step_key] = value
    await state.update_data(register_data=register_data, register_waiting=None)

    # برو به مرحله بعد
    idx = data.get("register_index", 0)
    await state.update_data(register_index=idx + 1)
    await state.set_state(RegisterMarketStates.flow)
    await _show_register_prompt(message, state)


@router.message(RegisterMarketStates.awaiting_value, Command("cancel"))
async def register_cancel_command(message: types.Message, state: FSMContext):
    await _cancel_register_flow(message, state)


@router.callback_query(RegisterMarketStates.flow, F.data == "reg:back")
async def register_back(callback: types.CallbackQuery, state: FSMContext):
    data = await state.get_data()
    idx = max(data.get("register_index", 0) - 1, 0)
    await state.update_data(register_index=idx)
    await callback.answer()
    await callback.message.answer("به مرحله قبل بازگشتید.")
    await _show_register_prompt(callback, state)


@router.callback_query(RegisterMarketStates.awaiting_value, F.data == "reg:back")
async def register_back_from_input(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(RegisterMarketStates.flow)
    await state.update_data(register_waiting=None)
    await callback.answer()
    await _show_register_prompt(callback, state)


@router.callback_query(F.data == "reg:cancel")
async def register_cancel(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer()
    await _cancel_register_flow(callback, state)


@router.callback_query(RegisterMarketStates.flow, F.data.startswith("reg:tier:"))
async def register_set_tier(callback: types.CallbackQuery, state: FSMContext):
    tier = callback.data.split(":")[-1]
    data = await state.get_data()
    register_data = data.get("register_data", {})
    register_data["subscription_tier"] = tier
    await state.update_data(register_data=register_data, register_index=len(REGISTER_FLOW))
    await callback.answer(f"اشتراک {tier} انتخاب شد.")
    await _show_register_prompt(callback, state)


@router.callback_query(RegisterMarketStates.flow, F.data == "reg:confirm")
async def register_confirm(callback: types.CallbackQuery, state: FSMContext):
    """تأیید نهایی ثبت مارکت جدید توسط هایپر ادمین."""
    await callback.answer()

    data = await state.get_data()
    register_data: dict[str, Any] = data.get("register_data", {}) or {}

    payload = {
        "title": register_data.get("title"),
        "store_admin_phone": register_data.get("store_admin_phone"),
        "address": register_data.get("address"),
        "category": register_data.get("category"),
        "bot_token": register_data.get("bot_token"),
        "work_hours": register_data.get("work_hours"),
        "subscription_tier": register_data.get("subscription_tier", "base"),
    }

    client = await _get_client(state)
    waiting = await callback.message.answer("⌛ در حال ثبت مارکت...")

    try:
        resp = await client.post("/tenants", payload)
        tenant_id = resp["id"]

        channel_payload: dict[str, Any] = {}
        if register_data.get("channel_id"):
            channel_payload["channel_id"] = register_data["channel_id"]
        if register_data.get("channel_invite_link"):
            channel_payload["channel_invite_link"] = register_data["channel_invite_link"]

        if channel_payload:
            await client.put(f"/tenants/{tenant_id}/profile", channel_payload)

    except Exception as exc:

        logger.exception("register_confirm failed", exc_info=exc)
        try:
            await waiting.edit_text(f"ثبت مارکت ناموفق بود: {exc}")
        except Exception:
            await callback.message.answer(f"ثبت مارکت ناموفق بود: {exc}")
        return

    success_text = (
        "✅ مارکت با موفقیت ایجاد شد.\n"
        f"کد مارکت: {resp['code']}\n"
        f"نام کاربری مدیر: {resp['admin_credentials']['username']}"
    )

    try:
        await waiting.edit_text(success_text)
    except Exception:
        await callback.message.answer(success_text)

    await callback.message.answer("به منوی مدیریت هایپر برگشتی. از گزینه‌های زیر ادامه بده:",reply_markup=keyboards.hyper_admin_menu(),)

    await state.set_state(AdminStates.menu)
    await _reset_register_flow(state)

# ---------------------------------------------------------------------------
# مدیریت مارکت‌ها (نمایش لیست و خلاصه)
# ---------------------------------------------------------------------------


@router.message(AdminStates.menu, F.text == "📂 مدیریت مارکت‌ها")
async def tenants_manage(message: types.Message, state: FSMContext):
    client = await _get_client(state)
    tenants = await client.get("/tenants")
    filtered = [t for t in tenants if t.get("code") != "hyper"]
    if not filtered:
        await message.answer("هیچ مارکت فعالی برای مدیریت وجود ندارد.")
        return
    await state.set_state(ManageTenantStates.select)
    await message.answer("مارکت مورد نظر را انتخاب کنید:", reply_markup=keyboards.tenants_keyboard(filtered))


def _format_tenant_summary(tenant: dict[str, Any]) -> str:
    status_map = {"active": "فعال", "inactive": "غیرفعال"}
    return "\n".join(
        [
            f"مارکت: {tenant.get('title')}",
            f"کد: {tenant.get('code')}",
            f"وضعیت: {status_map.get(tenant.get('status'), 'نامشخص')}",
            f"اشتراک: {tenant.get('subscription_tier')}",
            f"دسته‌بندی: {tenant.get('category') or '-'}",
            f"آدرس: {tenant.get('address') or '-'}",
            f"تماس: {tenant.get('contact_phone') or '-'}",
            f"ساعات پاسخ‌گویی: {tenant.get('work_hours') or '-'}",
        ]
    )


@router.callback_query(F.data == "tenant:back")
async def tenants_back(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(AdminStates.menu)
    await callback.answer()
    await callback.message.answer("به منوی اصلی بازگشتید.", reply_markup=keyboards.hyper_admin_menu())


@router.callback_query(F.data == "tenant:list")
async def tenants_list(callback: types.CallbackQuery, state: FSMContext):
    client = await _get_client(state)
    tenants = await client.get("/tenants")
    filtered = [t for t in tenants if t.get("code") != "hyper"]
    await callback.answer()
    await callback.message.answer("مارکت مورد نظر را انتخاب کنید:", reply_markup=keyboards.tenants_keyboard(filtered))


@router.callback_query(F.data == "tenant:cancel")
async def tenants_cancel(callback: types.CallbackQuery, state: FSMContext):
    await state.set_state(AdminStates.menu)
    await callback.answer()
    await callback.message.answer("به منوی اصلی بازگشتید.", reply_markup=keyboards.hyper_admin_menu())


@router.callback_query(F.data.startswith("tenant:"))
async def tenant_selected(callback: types.CallbackQuery, state: FSMContext):
    if callback.data in {"tenant:back", "tenant:list", "tenant:cancel"}:
        return
    tenant_id = callback.data.split(":", 1)[1]
    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    await state.update_data(selected_tenant_id=tenant_id)
    summary = _format_tenant_summary(tenant)
    await callback.answer()
    await callback.message.answer(
        summary,
        reply_markup=keyboards.tenant_actions_keyboard(tenant_id, tenant.get("status") == "active"),
    )


@router.callback_query(F.data.startswith("tact:toggle:"))
async def tenant_toggle(callback: types.CallbackQuery, state: FSMContext):
    tenant_id = callback.data.split(":")[-1]
    client = await _get_client(state)
    tenant = await client.get(f"/tenants/{tenant_id}")
    new_status = "inactive" if tenant.get("status") == "active" else "active"
    await client.put(f"/tenants/{tenant_id}", {"status": new_status})
    refreshed = await client.get(f"/tenants/{tenant_id}")
    summary = _format_tenant_summary(refreshed)
    await callback.answer("وضعیت به‌روزرسانی شد.")
    await callback.message.answer(
        summary,
        reply_markup=keyboards.tenant_actions_keyboard(tenant_id, refreshed.get("status") == "active"),
    )


@router.callback_query(F.data.startswith("tact:notify:"))
async def tenant_notify(callback: types.CallbackQuery, state: FSMContext):
    tenant_id = callback.data.split(":")[-1]
    client = await _get_client(state)
    try:
        await client.post(f"/tenants/{tenant_id}/notify", {"message": "کاربر مدیریتی جدید افزوده شد."})
    except Exception as exc:
        await callback.answer(f"ارسال اعلان ناموفق بود: {exc}", show_alert=True)
        return
    await callback.answer("اعلان ارسال شد.", show_alert=True)


@router.callback_query(F.data.startswith("tact:add_admin:"))
async def tenant_add_admin(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer("مدیریت مدیران به‌زودی اضافه می‌شود.", show_alert=True)


@router.callback_query(F.data.startswith("tact:edit_profile:"))
async def tenant_edit_profile(callback: types.CallbackQuery, state: FSMContext):
    await callback.answer("ویرایش پروفایل در حال توسعه است.", show_alert=True)
