#!/usr/bin/env python3
from __future__ import annotations
import os
import sys
from pathlib import Path

PROJECT_ROOT = Path(__file__).resolve().parent.parent
sys.path.insert(0, str(PROJECT_ROOT))
os.chdir(PROJECT_ROOT)

from sqlalchemy import select

from app.core.config import get_settings
from app.core.security import hash_password
from app.db.session import SessionLocal
from app.models import AppUser


def main() -> None:
    settings = get_settings()
    username = settings.DEFAULT_HYPER_ADMIN_USERNAME
    password = settings.DEFAULT_HYPER_ADMIN_PASSWORD
    if not username or not password:
        raise SystemExit("DEFAULT_HYPER_ADMIN_USERNAME and DEFAULT_HYPER_ADMIN_PASSWORD must be set in the environment")

    password_hash = hash_password(password)
    with SessionLocal() as db:
        stmt = select(AppUser).where(AppUser.role == "hyper_admin")
        admins = list(db.scalars(stmt))
        if not admins:
            raise SystemExit("no hyper_admin account found in the database")
        for admin in admins:
            admin.login_username = username
            admin.password_hash = password_hash
            admin.status = "active"
            db.add(admin)
        db.commit()
    print(f"Updated {len(admins)} hyper_admin account(s) to `{username}`")


if __name__ == "__main__":
    main()
