from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy.orm import Session
from sqlalchemy.exc import IntegrityError
from sqlalchemy import select
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.schemas.product import ProductCreate, ProductUpdate, ProductImageIn, SpecIn
from app.services import catalog as catalog_service
from app.models import Product, Category


router = APIRouter(prefix="/products", tags=["products"])


def _ensure_product(db: Session, tenant_id: str, product_id: str) -> Product:
    product = db.get(Product, product_id)
    if not product or str(product.tenant_id) != str(tenant_id):
        raise HTTPException(status_code=404, detail="product not found")
    return product


@router.get("/", response_model=list[dict])
def list_products(
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    category_id: str | None = Query(default=None),
    active: bool | None = Query(default=None),
    q: str | None = Query(default=None, description="search by title/sku"),
):
    stmt = select(Product).where(Product.tenant_id == tenant.id)
    if category_id:
        stmt = stmt.join(Product.categories).where(Category.id == category_id)
    if active is not None:
        stmt = stmt.where(Product.active == active)
    if q:
        pattern = f"%{q.lower()}%"
        stmt = stmt.where(Product.sku.ilike(pattern) | Product.title.ilike(pattern))
    stmt = stmt.distinct().order_by(Product.created_at.desc())
    records = db.scalars(stmt).all()
    return [catalog_service.serialize_product(prod) for prod in records]


@router.get("/{product_id}", response_model=dict)
def get_product(product_id: str, tenant=Depends(get_tenant_context), db: Session = Depends(get_db)):
    product = _ensure_product(db, str(tenant.id), product_id)
    return catalog_service.serialize_product(product)


@router.post("/", response_model=dict)
def create_product(
    payload: ProductCreate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    try:
        product = catalog_service.create_product(db, str(tenant.id), payload)
        return catalog_service.serialize_product(product)
    except IntegrityError:
        db.rollback()
        raise HTTPException(status_code=409, detail="SKU already exists for this tenant")
    except ValueError as e:
        db.rollback()
        raise HTTPException(status_code=400, detail=str(e))

@router.put("/{product_id}", response_model=dict)
def update_product(
    product_id: str,
    payload: ProductUpdate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    product = _ensure_product(db, str(tenant.id), product_id)
    product = catalog_service.update_product(db, product, payload)
    return catalog_service.serialize_product(product)


@router.post("/{product_id}/images", response_model=dict)
def add_product_image_endpoint(
    product_id: str,
    payload: ProductImageIn,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    product = _ensure_product(db, str(tenant.id), product_id)
    catalog_service.add_product_image(db, product, payload)
    return catalog_service.serialize_product(product)


@router.post("/{product_id}/specs", response_model=dict)
def add_product_spec_endpoint(
    product_id: str,
    payload: SpecIn,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    product = _ensure_product(db, str(tenant.id), product_id)
    catalog_service.add_product_spec(db, product, payload)
    return catalog_service.serialize_product(product)


@router.delete("/{product_id}/specs/{spec_id}")
def delete_spec_endpoint(
    product_id: str,
    spec_id: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    product = _ensure_product(db, str(tenant.id), product_id)
    catalog_service.delete_product_spec(db, product, spec_id)
    return {"deleted": True}


@router.get("/{product_id}/price-history", response_model=list[dict])
def price_history(
    product_id: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    entries = catalog_service.list_price_history(db, str(tenant.id), product_id)
    if not entries:
        _ensure_product(db, str(tenant.id), product_id)
    return catalog_service.serialize_price_history(entries)
