from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session
from app.api.dependencies import get_db, get_tenant_context, require_roles
from app.schemas.category import CategoryCreate, CategoryUpdate, CategoryOut
from app.services import catalog as catalog_service
from app.models import Category


router = APIRouter(prefix="/categories", tags=["categories"])


@router.get("/", response_model=list[CategoryOut])
def list_categories(
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
):
    categories = catalog_service.list_categories(db, str(tenant.id))
    return [CategoryOut(id=c.id, name=c.name, parent_id=c.parent_id) for c in categories]


@router.post("/", response_model=CategoryOut)
def create_category(
    payload: CategoryCreate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    category = catalog_service.create_category(db, str(tenant.id), payload)
    return CategoryOut(id=category.id, name=category.name, parent_id=category.parent_id)


@router.put("/{category_id}", response_model=CategoryOut)
def update_category(
    category_id: str,
    payload: CategoryUpdate,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    category = db.get(Category, category_id)
    if not category or str(category.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="category not found")
    category = catalog_service.update_category(db, category, payload)
    return CategoryOut(id=category.id, name=category.name, parent_id=category.parent_id)


@router.delete("/{category_id}")
def delete_category(
    category_id: str,
    tenant=Depends(get_tenant_context),
    db: Session = Depends(get_db),
    _: object = Depends(require_roles("store_admin", "hyper_admin")),
):
    category = db.get(Category, category_id)
    if not category or str(category.tenant_id) != str(tenant.id):
        raise HTTPException(status_code=404, detail="category not found")
    catalog_service.delete_category(db, category)
    return {"deleted": True}
