from __future__ import annotations
import uuid
from datetime import datetime
from typing import Sequence
from sqlalchemy import select
from sqlalchemy.orm import Session
from app.models import StoreVisit, ProductLike, Product


def _ensure_uuid(value: str | uuid.UUID) -> uuid.UUID:
    return value if isinstance(value, uuid.UUID) else uuid.UUID(str(value))


def get_visit_by_token(db: Session, tenant_id: str, session_token: str) -> StoreVisit | None:
    stmt = select(StoreVisit).where(
        StoreVisit.session_token == session_token,
        StoreVisit.tenant_id == _ensure_uuid(tenant_id),
    )
    return db.scalars(stmt).first()


def get_active_visit(db: Session, tenant_id: str, telegram_user_id: int) -> StoreVisit | None:
    stmt = select(StoreVisit).where(
        StoreVisit.tenant_id == _ensure_uuid(tenant_id),
        StoreVisit.telegram_user_id == telegram_user_id,
        StoreVisit.status == "active",
    )
    return db.scalars(stmt).first()


def start_visit(db: Session, tenant_id: str, telegram_user_id: int) -> StoreVisit:
    existing = get_active_visit(db, tenant_id, telegram_user_id)
    if existing:
        return existing
    visit = StoreVisit(
        tenant_id=_ensure_uuid(tenant_id),
        telegram_user_id=telegram_user_id,
        status="active",
        started_at=datetime.utcnow(),
    )
    db.add(visit)
    db.commit()
    db.refresh(visit)
    return visit


def end_visit(db: Session, visit: StoreVisit, status: str = "completed") -> StoreVisit:
    visit.status = status
    visit.ended_at = datetime.utcnow()
    db.add(visit)
    db.commit()
    db.refresh(visit)
    return visit


def record_like(
    db: Session,
    tenant_id: str,
    product_id: str,
    telegram_user_id: int,
    session: StoreVisit | None = None,
) -> ProductLike:
    product = db.get(Product, _ensure_uuid(product_id))
    if not product or str(product.tenant_id) != str(tenant_id):
        raise ValueError("product not found")
    like = db.scalars(
        select(ProductLike).where(
            ProductLike.tenant_id == _ensure_uuid(tenant_id),
            ProductLike.product_id == product.id,
            ProductLike.telegram_user_id == telegram_user_id,
        )
    ).first()
    if like:
        like.liked_at = datetime.utcnow()
        if session:
            like.session_id = session.id
    else:
        like = ProductLike(
            tenant_id=_ensure_uuid(tenant_id),
            product_id=product.id,
            telegram_user_id=telegram_user_id,
            session_id=session.id if session else None,
        )
        db.add(like)
    db.commit()
    db.refresh(like)
    return like


def list_likes_for_session(db: Session, session: StoreVisit) -> Sequence[ProductLike]:
    stmt = (
        select(ProductLike)
        .where(ProductLike.session_id == session.id)
        .order_by(ProductLike.liked_at.asc())
    )
    return db.scalars(stmt).all()


def list_likes_between(db: Session, tenant_id: str, telegram_user_id: int, since: datetime, until: datetime):
    stmt = (
        select(ProductLike)
        .where(
            ProductLike.tenant_id == _ensure_uuid(tenant_id),
            ProductLike.telegram_user_id == telegram_user_id,
            ProductLike.liked_at >= since,
            ProductLike.liked_at <= until,
        )
        .order_by(ProductLike.liked_at)
    )
    return db.scalars(stmt).all()


def list_likes_for_user(db: Session, tenant_id: str, telegram_user_id: int) -> Sequence[ProductLike]:
    stmt = (
        select(ProductLike)
        .where(
            ProductLike.tenant_id == _ensure_uuid(tenant_id),
            ProductLike.telegram_user_id == telegram_user_id,
        )
        .order_by(ProductLike.liked_at.desc())
    )
    return db.scalars(stmt).all()


def delete_like(db: Session, tenant_id: str, telegram_user_id: int, product_id: str) -> bool:
    like = db.scalars(
        select(ProductLike).where(
            ProductLike.tenant_id == _ensure_uuid(tenant_id),
            ProductLike.telegram_user_id == telegram_user_id,
            ProductLike.product_id == _ensure_uuid(product_id),
        )
    ).first()
    if not like:
        return False
    db.delete(like)
    db.commit()
    return True
