from __future__ import annotations

import json
import sqlite3
from datetime import datetime, date, time as dtime
from pathlib import Path
from typing import Optional

from bot.models import Board, Column, Task, TaskEvent, StandupConfig, StandupEntry, BoardMember


BASE_DIR = Path(__file__).resolve().parents[2]  # .../src
DB_PATH = BASE_DIR / "bale_team_bot.db"


class SQLiteBoardRepository:
    """ریپازیتوری مبتنی بر SQLite برای ذخیره‌سازی بردها، تسک‌ها و اعضا."""

    def __init__(self, db_path: Optional[str | Path] = None) -> None:
        self.db_path = str(db_path or DB_PATH)
        self._ensure_schema()

    def _connect(self) -> sqlite3.Connection:
        conn = sqlite3.connect(self.db_path)
        conn.row_factory = sqlite3.Row
        return conn

    def _ensure_schema(self) -> None:
        conn = self._connect()
        cur = conn.cursor()
        cur.executescript(
            """
            CREATE TABLE IF NOT EXISTS boards (
                chat_id INTEGER PRIMARY KEY
            );

            CREATE TABLE IF NOT EXISTS board_columns (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                chat_id INTEGER NOT NULL,
                col_key TEXT NOT NULL,
                name TEXT NOT NULL,
                order_index INTEGER NOT NULL,
                wip_limit INTEGER,
                is_done_column INTEGER NOT NULL DEFAULT 0,
                UNIQUE(chat_id, col_key)
            );

            CREATE TABLE IF NOT EXISTS tasks (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                chat_id INTEGER NOT NULL,
                title TEXT NOT NULL,
                column_key TEXT NOT NULL,
                creator_id INTEGER NOT NULL,
                assignee_id INTEGER,
                created_at TEXT NOT NULL,
                done_at TEXT,
                class_of_service TEXT NOT NULL,
                due_at TEXT,
                is_blocked INTEGER NOT NULL DEFAULT 0,
                blocked_reason TEXT
            );

            CREATE TABLE IF NOT EXISTS task_events (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                chat_id INTEGER NOT NULL,
                task_id INTEGER NOT NULL,
                from_column TEXT,
                to_column TEXT,
                actor_id INTEGER,
                created_at TEXT NOT NULL,
                event_type TEXT NOT NULL,
                extra_json TEXT
            );

            CREATE TABLE IF NOT EXISTS standup_configs (
                chat_id INTEGER PRIMARY KEY,
                enabled INTEGER NOT NULL,
                time_of_day TEXT,
                questions_json TEXT NOT NULL
            );

            CREATE TABLE IF NOT EXISTS standup_entries (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                chat_id INTEGER NOT NULL,
                user_id INTEGER NOT NULL,
                standup_date TEXT NOT NULL,
                answers_json TEXT NOT NULL,
                created_at TEXT NOT NULL
            );

            CREATE TABLE IF NOT EXISTS board_members (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                chat_id INTEGER NOT NULL,
                user_id INTEGER NOT NULL,
                display_name TEXT,
                role TEXT NOT NULL,
                is_owner INTEGER NOT NULL DEFAULT 0,
                UNIQUE(chat_id, user_id)
            );
            """
        )
        conn.commit()
        conn.close()

    def get_board(self, chat_id: int) -> Board:
        conn = self._connect()
        cur = conn.cursor()

        cur.execute("INSERT OR IGNORE INTO boards(chat_id) VALUES (?)", (chat_id,))
        conn.commit()

        board = Board(chat_id=chat_id)

        # columns
        cur.execute(
            "SELECT col_key, name, order_index, wip_limit, is_done_column "
            "FROM board_columns WHERE chat_id=? ORDER BY order_index",
            (chat_id,),
        )
        col_rows = cur.fetchall()
        if col_rows:
            for r in col_rows:
                board.columns.append(
                    Column(
                        key=r["col_key"],
                        name=r["name"],
                        order_index=r["order_index"],
                        wip_limit=r["wip_limit"],
                        is_done_column=bool(r["is_done_column"]),
                    )
                )
        else:
            board.ensure_default_columns()

        # tasks
        cur.execute("SELECT * FROM tasks WHERE chat_id=?", (chat_id,))
        task_rows = cur.fetchall()
        max_id = 0
        for r in task_rows:
            tid = r["id"]
            max_id = max(max_id, tid)
            created_at = datetime.fromisoformat(r["created_at"])
            done_at = datetime.fromisoformat(r["done_at"]) if r["done_at"] else None
            due_at = datetime.fromisoformat(r["due_at"]) if r["due_at"] else None
            task = Task(
                id=tid,
                title=r["title"],
                column_key=r["column_key"],
                creator_id=r["creator_id"],
                assignee_id=r["assignee_id"],
                created_at=created_at,
                done_at=done_at,
                class_of_service=r["class_of_service"],
                due_at=due_at,
                is_blocked=bool(r["is_blocked"]),
                blocked_reason=r["blocked_reason"],
            )
            board.tasks[tid] = task
        board.next_task_id = max_id + 1 if max_id > 0 else 1

        # events
        cur.execute("SELECT * FROM task_events WHERE chat_id=?", (chat_id,))
        ev_rows = cur.fetchall()
        for r in ev_rows:
            extra = json.loads(r["extra_json"]) if r["extra_json"] else {}
            ev = TaskEvent(
                task_id=r["task_id"],
                from_column=r["from_column"],
                to_column=r["to_column"],
                actor_id=r["actor_id"],
                created_at=datetime.fromisoformat(r["created_at"]),
                event_type=r["event_type"],
                extra=extra,
            )
            board.events.append(ev)

        # standup config
        cur.execute("SELECT * FROM standup_configs WHERE chat_id=?", (chat_id,))
        row = cur.fetchone()
        if row:
            time_of_day = None
            if row["time_of_day"]:
                hh, mm = map(int, row["time_of_day"].split(":", 1))
                time_of_day = dtime(hour=hh, minute=mm)
            questions = json.loads(row["questions_json"]) if row["questions_json"] else []
            cfg = StandupConfig(
                team_chat_id=chat_id,
                enabled=bool(row["enabled"]),
                time_of_day=time_of_day,
                questions=questions,
            )
            board.standup_config = cfg

        # standup entries
        cur.execute("SELECT * FROM standup_entries WHERE chat_id=?", (chat_id,))
        ent_rows = cur.fetchall()
        for r in ent_rows:
            entry = StandupEntry(
                team_chat_id=chat_id,
                user_id=r["user_id"],
                standup_date=date.fromisoformat(r["standup_date"]),
                answers=json.loads(r["answers_json"]),
                created_at=datetime.fromisoformat(r["created_at"]),
            )
            board.standup_entries.append(entry)

        # members
        cur.execute("SELECT * FROM board_members WHERE chat_id=?", (chat_id,))
        mem_rows = cur.fetchall()
        for r in mem_rows:
            m = BoardMember(
                user_id=r["user_id"],
                display_name=r["display_name"] or "",
                role=r["role"],
                is_owner=bool(r["is_owner"]),
            )
            board.members[m.user_id] = m

        conn.close()
        return board

    def save_board(self, board: Board) -> None:
        conn = self._connect()
        cur = conn.cursor()

        with conn:
            # columns
            cur.execute("DELETE FROM board_columns WHERE chat_id=?", (board.chat_id,))
            for c in board.columns:
                cur.execute(
                    """
                    INSERT INTO board_columns (chat_id, col_key, name, order_index, wip_limit, is_done_column)
                    VALUES (?, ?, ?, ?, ?, ?)
                    """,
                    (board.chat_id, c.key, c.name, c.order_index, c.wip_limit, int(c.is_done_column)),
                )

            # tasks
            cur.execute("DELETE FROM tasks WHERE chat_id=?", (board.chat_id,))
            for t in board.tasks.values():
                cur.execute(
                    """
                    INSERT INTO tasks (
                        id, chat_id, title, column_key, creator_id, assignee_id,
                        created_at, done_at, class_of_service, due_at,
                        is_blocked, blocked_reason
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    """,
                    (
                        t.id,
                        board.chat_id,
                        t.title,
                        t.column_key,
                        t.creator_id,
                        t.assignee_id,
                        t.created_at.isoformat(),
                        t.done_at.isoformat() if t.done_at else None,
                        t.class_of_service,
                        t.due_at.isoformat() if t.due_at else None,
                        int(t.is_blocked),
                        t.blocked_reason,
                    ),
                )

            # events
            cur.execute("DELETE FROM task_events WHERE chat_id=?", (board.chat_id,))
            for e in board.events:
                cur.execute(
                    """
                    INSERT INTO task_events (
                        chat_id, task_id, from_column, to_column,
                        actor_id, created_at, event_type, extra_json
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    """,
                    (
                        board.chat_id,
                        e.task_id,
                        e.from_column,
                        e.to_column,
                        e.actor_id,
                        e.created_at.isoformat(),
                        e.event_type,
                        json.dumps(e.extra) if e.extra else None,
                    ),
                )

            # standup config
            if board.standup_config is not None:
                cfg = board.standup_config
                time_str = cfg.time_of_day.strftime("%H:%M") if cfg.time_of_day else None
                cur.execute(
                    """
                    INSERT INTO standup_configs (chat_id, enabled, time_of_day, questions_json)
                    VALUES (?, ?, ?, ?)
                    ON CONFLICT(chat_id) DO UPDATE SET
                        enabled=excluded.enabled,
                        time_of_day=excluded.time_of_day,
                        questions_json=excluded.questions_json
                    """,
                    (
                        board.chat_id,
                        int(cfg.enabled),
                        time_str,
                        json.dumps(cfg.questions),
                    ),
                )
            else:
                cur.execute("DELETE FROM standup_configs WHERE chat_id=?", (board.chat_id,))

            # standup entries
            cur.execute("DELETE FROM standup_entries WHERE chat_id=?", (board.chat_id,))
            for e in board.standup_entries:
                cur.execute(
                    """
                    INSERT INTO standup_entries (
                        chat_id, user_id, standup_date, answers_json, created_at
                    ) VALUES (?, ?, ?, ?, ?)
                    """,
                    (
                        board.chat_id,
                        e.user_id,
                        e.standup_date.isoformat(),
                        json.dumps(e.answers),
                        e.created_at.isoformat(),
                    ),
                )

            # members
            cur.execute("DELETE FROM board_members WHERE chat_id=?", (board.chat_id,))
            for m in board.members.values():
                cur.execute(
                    """
                    INSERT INTO board_members (
                        chat_id, user_id, display_name, role, is_owner
                    ) VALUES (?, ?, ?, ?, ?)
                    """,
                    (
                        board.chat_id,
                        m.user_id,
                        m.display_name,
                        m.role,
                        int(m.is_owner),
                    ),
                )

        conn.close()
